package db

import (
	"strconv"
	"time"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/common/stats"

	es "github.com/Kaidence/elastigo/lib"
)

const (
	MaxConcurrentRequests = 20
)

type ElasticSearchWriter struct {
	Conn    *es.Conn
	Indexer *es.BulkIndexer
	Done    chan bool
	Index   string
	Type    string
}

// NewElasticSearchWriter creates a new connection to ElasticSearch meant for writing data.
func NewElasticSearchWriter(conf *config.Config) *ElasticSearchWriter {
	client := &ElasticSearchWriter{
		Conn:  es.NewConn(),
		Index: conf.ESIndex,
		Type:  conf.ESType,
	}
	client.Conn.SetHosts(conf.ESHosts)
	client.Conn.SetPort(strconv.Itoa(conf.ESPort))

	// Set up the BulkIndexer's for sending updates
	client.Indexer = client.Conn.NewBulkIndexer(MaxConcurrentRequests, true)
	client.Indexer.Start()

	go func() {
		for errBuffer := range client.Indexer.ErrorChannel {
			log.Reportf("failed to send update to elasticsearch: %s: %s", errBuffer.Err, errBuffer.Buf.String())
		}
	}()

	stat := stats.InitStatsd(conf)
	go func() {
		for {
			stat.Gauge("es.bulk.pending", int64(client.Indexer.PendingDocuments()), 1)
			time.Sleep(1 * time.Second)
		}
	}()

	go func() {
		if client.Indexer.StatsCh != nil {
			for esStats := range client.Indexer.StatsCh {
				stat.TimingDuration("es.bulk.index", esStats.Duration, 1)
				stat.Inc("es.bulk.errors", int64(esStats.NumErrors), 1)
			}
		}
	}()

	return client
}

// Update takes information about a single channel from a specific source, and updates
// the entry in ElasticSearch.
func (T *ElasticSearchWriter) Update(channel string, input map[string]interface{}) *JaxDbError {
	// ElasticSearch doesn't like empty bodies
	if input == nil {
		input = map[string]interface{}{}
	}

	m := map[string]interface{}{"doc": input, "doc_as_upsert": true}

	return formatError(T.Indexer.Update(T.Index, T.Type, channel, "", nil, m, false))
}

// Delete deletes a channel from ES.
func (T *ElasticSearchWriter) Delete(channel string) {
	T.Indexer.Delete(T.Index, T.Type, channel, false)
}

// CreateIndexIfNotExists creates the index required for Jax if it doesn't already exist.
// Requires shard/replica settings.
func (T *ElasticSearchWriter) CreateIndexIfNotExists(shards, replicas int) *JaxDbError {
	exists, err := T.Conn.IndicesExists(T.Index)
	if err != nil {
		log.Printf("failed to check for existence of index. attempting to create...\n")
	}
	if !exists {
		_, err := T.Conn.CreateIndexWithSettings(T.Index, mappingSettings(T.Type, shards, replicas))

		if err != nil {
			log.Reportf("Failed to create index %s", err.Error())
		}
	}

	return nil
}

func mappingSettings(_type string, shards, replicas int) IndexSettings {
	settings := map[string]interface{}{
		"index": map[string]interface{}{
			"analysis": map[string]interface{}{
				"analyzer": map[string]interface{}{
					"default": map[string]interface{}{
						"tokenizer": "keyword",
						"filter":    "lowercase",
					},
				},
			},
			"number_of_shards":   shards,
			"number_of_replicas": replicas,
		},
	}
	return IndexSettings{Settings: settings}
}

// IndexSettings represents the initial settings for the "jax" index.
type IndexSettings struct {
	Settings map[string]interface{} `json:"settings"`
}
