package query

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/web/jax/common/language"
)

type AggregationQuery struct {
	Limit   int
	Offset  int
	GroupBy string
	Locale  string
	Sort    string
	Filters []Filter
}

var aggQuery = map[string]interface{}{
	"filtered": map[string]interface{}{
		"filter": AndFilter(ExistsFieldFilter("usher.channel_count")),
	},
}

func (A AggregationQuery) ToQuery() []byte {
	var sort string
	if A.Sort != "" {
		sort = A.Sort
	} else if A.Locale != "" && language.ValidLocales[A.Locale] {
		sort = fmt.Sprintf("usher.weighted_language_ccu.%v", A.Locale)
	}

	filters := append(A.Filters, ExistsFieldFilter("usher.channel_count"))
	if sort != "" {
		filters = append(filters, ExistsFieldFilter(sort))
	}

	q := map[string]interface{}{
		"query": map[string]interface{}{
			"filtered": map[string]interface{}{
				"filter": AndFilter(filters...),
			},
		},
		"size":         0,
		"aggregations": AggregationSubQuery(A.GroupBy, A.Limit, sort),
	}

	b, err := json.Marshal(q)
	if err != nil {
		return []byte{}
	}

	return b
}

func AggregationSubQuery(groupBy string, limit int, sort string) map[string]interface{} {
	aggregations := map[string]interface{}{
		"total_count": map[string]interface{}{
			"sum": map[string]interface{}{
				"field": "usher.channel_count",
			},
		},
	}

	// which sum aggregation to sort by
	sortBy := "total_count"

	if sort != "" {
		aggregations["alt_sort_count"] = map[string]interface{}{
			"sum": map[string]interface{}{
				"field": sort,
			},
		}
		sortBy = "alt_sort_count"
	}

	return map[string]interface{}{
		"results": map[string]interface{}{
			"terms": map[string]interface{}{
				"field": groupBy,
				"order": map[string]interface{}{
					sortBy: "desc",
				},
				"size": limit,
			},
			"aggregations": aggregations,
		},
	}
}
