package query

import (
	"encoding/json"
	"fmt"
	"strconv"
	"strings"
	"time"
)

const (
	ttl         = "3m"
	ttlDuration = 3 * time.Minute
)

type Filter interface {
	Valid(properties map[string]interface{}) bool
}

type EmptyFilter struct{}

func (F EmptyFilter) Valid(properties map[string]interface{}) bool {
	return true
}

type FilterImpl map[string]interface{}

func (F FilterImpl) Valid(properties map[string]interface{}) bool {
	for k, v := range F {
		switch k {
		case "term":
			inner, ok := v.(map[string]interface{})
			if ok {
				for key, value := range inner {
					if strings.ToLower(fmt.Sprintf("%v", properties[key])) == strings.ToLower(fmt.Sprintf("%v", value)) {
						return true
					}
				}
				return false
			}
		case "terms":
			inner, ok := v.(map[string]interface{})
			if ok {
				for key, value := range inner {
					if values, ok := value.([]string); ok {
						for _, val := range values {
							if strings.ToLower(fmt.Sprintf("%v", properties[key])) == strings.ToLower(fmt.Sprintf("%v", val)) {
								return true
							}
						}
					}
				}
				return false
			}
		case "range":
			inner, ok := v.(map[string]interface{})
			if ok {
				for key, value := range inner {
					if innerInner, ok := value.(map[string]interface{}); ok {
						for kind, val := range innerInner {
							res, err := compareFloatStrings(kind, fmt.Sprintf("%v", properties[key]), fmt.Sprintf("%v", val))
							if err == nil && res {
								continue
							}

							res, err = compareTimeStrings(kind, fmt.Sprintf("%v", properties[key]), fmt.Sprintf("%v", val))
							if err == nil && !res {
								continue
							}

							return false
						}
						return true
					}
				}
				return false
			}
			return true
		case "exists":
			inner, ok := v.(map[string]interface{})
			if ok {
				if field, ok := inner["field"].(string); ok {
					if _, ok := properties[field]; ok {
						return true
					}
				}
				return false
			}
		case "not":
			inner, ok := v.(Filter)
			if ok {
				return !inner.Valid(properties)
			}
		case "or":
			inner, ok := v.([]Filter)
			if ok {
				for _, f := range inner {
					if f.Valid(properties) {
						return true
					}
				}
				return false
			}
		default:
			continue
		}
	}

	return true
}

func compareFloatStrings(kind, key, val string) (bool, error) {
	keyVal, err := strconv.ParseFloat(fmt.Sprintf("%v", key), 64)
	if err != nil {
		return false, err
	}
	rangeVal, err := strconv.ParseFloat(fmt.Sprintf("%v", val), 64)
	if err != nil {
		return false, err
	}

	switch kind {
	case "lt":
		return keyVal < rangeVal, nil
	case "lte":
		return keyVal <= rangeVal, nil
	case "gt":
		return keyVal > rangeVal, nil
	case "gte":
		return keyVal >= rangeVal, nil
	}
	return false, fmt.Errorf("invalid comparison %s", kind)
}

func compareTimeStrings(kind, key, val string) (bool, error) {
	keyTime, err := time.Parse(time.RFC3339Nano, key)
	if err != nil {
		return false, err
	}
	valTime, err := time.Parse(time.RFC3339Nano, val)
	if err != nil {
		return false, err
	}

	switch kind {
	case "lt":
		return keyTime.Before(valTime), nil
	case "lte":
		return keyTime.Before(valTime) || keyTime.Equal(valTime), nil
	case "gt":
		return keyTime.After(valTime), nil
	case "gte":
		return keyTime.After(valTime) || keyTime.Equal(valTime), nil
	}
	return false, fmt.Errorf("invalid comparison %s", kind)
}

// BodyToFilters converts an untyped map[string]interface{} into an array of Filters
func BodyToFilters(b map[string]interface{}) []Filter {
	filters := []Filter{}
	for key, val := range b {
		switch key {
		case "exists":
			if field, ok := val.(string); ok {
				filters = append(filters, ExistsFieldFilter(field))
			}
		case "eq":
			if e, ok := val.(map[string]interface{}); ok {
				for k, v := range e {
					if arr, ok := v.([]interface{}); ok {
						filters = append(filters, TermsFilter(k, arr))
					} else {
						filters = append(filters, TermFilter(k, v))
					}
				}
			}
		case "lt", "gt", "lte", "gte":
			if c, ok := val.(map[string]interface{}); ok {
				for k, v := range c {
					filters = append(filters, CompareFilter(key, k, v))
				}
			}
		case "and":
			if sub, ok := val.([]interface{}); ok {
				for _, v := range sub {
					if body, ok := v.(map[string]interface{}); ok {
						filters = append(filters, BodyToFilters(body)...)
					}
				}
			}
		case "or":
			subFilters := []Filter{}
			if sub, ok := val.([]interface{}); ok {
				for _, v := range sub {
					if body, ok := v.(map[string]interface{}); ok {
						subFilters = append(subFilters, BodyToFilters(body)...)
					}
				}
			}
			filters = append(filters, OrFilter(subFilters...))
		case "not":
			if c, ok := val.(map[string]interface{}); ok {
				subFilters := BodyToFilters(c)
				if len(subFilters) > 0 {
					filters = append(filters, NotFilter(subFilters[0]))
				}
			}
		}
	}
	return filters
}

// TTLAliveFilter is a filter on documents that are recently updated.
func TTLAliveFilter() Filter {
	return NotFilter(TTLExpiredFilter())
}

// TTLExpiredFilter is a filter on documents that have expired.
func TTLExpiredFilter() Filter {
	return OrFilter(
		MissingFieldFilter("streamlist.last_updated"),
		ExpiredFieldFilter("streamlist.last_updated", ttlDuration),
	)
}

// MissingFieldFilter is a filter on documents that don't have a field.
func MissingFieldFilter(field string) Filter {
	return NotFilter(ExistsFieldFilter(field))
}

// ExistsFieldFilter is a filter on documents that have a field.
func ExistsFieldFilter(field string) Filter {
	return FilterImpl{
		"exists": map[string]interface{}{
			"field": field,
		},
	}
}

type expiredFieldFilter struct {
	field    string
	duration time.Duration
}

func (F expiredFieldFilter) Valid(properties map[string]interface{}) bool {
	if val, ok := properties[F.field].(time.Time); ok {
		return val.Before(time.Now().Add(-F.duration))
	}
	if val, ok := properties[F.field].(string); ok {
		t, err := time.Parse(time.RFC3339Nano, val)

		if err != nil {
			return false
		}
		return t.Before(time.Now().Add(-F.duration))
	}
	return false
}

func (F expiredFieldFilter) MarshalJSON() ([]byte, error) {
	return json.Marshal(CompareFilter("lte", F.field, fmt.Sprintf("now-%vs", F.duration.Seconds())))
}

// ExpiredFieldFilter is a filter on documents with a field that is older than some expiry time.
func ExpiredFieldFilter(field string, expireTime time.Duration) Filter {
	return expiredFieldFilter{
		field:    field,
		duration: expireTime,
	}
}

// TermFilter is a filter on documents with a field with a specific value.
func TermFilter(field string, value interface{}) Filter {
	return FilterImpl{
		"term": map[string]interface{}{
			field: value,
		},
	}
}

// TermsFilter is a filter on documents with a field with any of an array of values.
func TermsFilter(field string, values []interface{}) Filter {
	if len(values) == 1 {
		return TermFilter(field, values[0])
	}
	return FilterImpl{
		"terms": map[string]interface{}{
			field: values,
		},
	}
}

// StringTermsFilter is a filter on documents with a field with any of an array of string values.
func StringTermsFilter(field string, values []string) Filter {
	if len(values) == 1 {
		return TermFilter(field, values[0])
	}
	return FilterImpl{
		"terms": map[string]interface{}{
			field: values,
		},
	}
}

// NotFilter is a filter that is the inverse of another filter.
func NotFilter(filter Filter) Filter {
	return FilterImpl{
		"not": filter,
	}
}

// OrFilter is a filter that is the OR of an array of filters.
func OrFilter(filters ...Filter) Filter {
	return FilterImpl{
		"or": filters,
	}
}

// AndFilter is a filter that is the AND of an array of filters.
func AndFilter(filters ...Filter) Filter {
	return FilterImpl{
		"and": filters,
	}
}

// CompareFilter is a filter on documents according to a comparison on a field.
func CompareFilter(kind, field string, value interface{}) Filter {
	return FilterImpl{
		"range": map[string]interface{}{
			field: map[string]interface{}{
				kind: value,
			},
		},
	}
}

func StreamTypeFilter(value string) Filter {
	switch value {
	case "live":
		return NotFilter(TermFilter("usher.broadcaster", "spectre"))
	case "playlist":
		return TermFilter("usher.broadcaster", "spectre")
	case "watch_party":
		return TermFilter("usher.broadcaster", "watch_party")
	default:
		return EmptyFilter{}
	}
}

func BroadcastPlatformFilter(value string) Filter {
	switch value {
	case "live":
		return NotFilter(OrFilter(TermFilter("usher.broadcaster", "spectre"), TermFilter("usher.broadcaster", "watch_party_rerun")))
	case "live_video":
		return NotFilter(OrFilter(TermFilter("usher.broadcaster", "watch_party"), TermFilter("usher.broadcaster", "spectre"), TermFilter("usher.broadcaster", "watch_party_premiere"), TermFilter("usher.broadcaster", "watch_party_rerun")))
	case "live_and_premieres":
		return NotFilter(OrFilter(TermFilter("usher.broadcaster", "watch_party"), TermFilter("usher.broadcaster", "spectre"), TermFilter("usher.broadcaster", "watch_party_rerun")))
	case "playlist":
		return TermFilter("usher.broadcaster", "spectre")
	case "watch_party":
		return TermFilter("usher.broadcaster", "watch_party")
	case "premiere":
		return TermFilter("usher.broadcaster", "watch_party_premiere")
	case "rerun":
		return TermFilter("usher.broadcaster", "watch_party_rerun")
	case "mobile":
		return StringTermsFilter("usher.broadcaster", []string{"ios", "android"})
	case "ps4":
		return TermFilter("usher.broadcaster", "octodad")
	case "xbox":
		return TermFilter("usher.broadcaster", "candybox")
	default:
		return EmptyFilter{}
	}
}

func XboxHeartbeatFilter() Filter {
	return OrFilter(TermFilter("usher.broadcaster", "candybox"), TermFilter("xbox_heartbeat.live", "true"))
}
