package query

import (
	"strconv"
	"strings"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

var (
	testField        = "blahh"
	testValues       = []string{"a", "b", "c"}
	testNumericValue = 245
)

func TestTermFilter(t *testing.T) {
	filter := TermFilter(testField, testValues[0])

	Convey("without the field present", t, func() {
		properties := map[string]interface{}{
			"foo": "hi",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and wrong value", t, func() {
		properties := map[string]interface{}{
			testField: "sdflkjdfkljskldfj",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and right value", t, func() {
		properties := map[string]interface{}{
			testField: testValues[0],
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})

	Convey("with the field present and right value but different case", t, func() {
		properties := map[string]interface{}{
			testField: strings.ToUpper(testValues[0]),
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})
}

func TestExistsFieldFilter(t *testing.T) {
	filter := ExistsFieldFilter(testField)

	Convey("without the field present", t, func() {
		properties := map[string]interface{}{
			"foo": "hi",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present", t, func() {
		properties := map[string]interface{}{
			testField: "sdflkjdfkljskldfj",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})
}

func TestStringTermsFilter(t *testing.T) {
	filter := StringTermsFilter(testField, testValues)

	Convey("without the field present", t, func() {
		properties := map[string]interface{}{
			"foo": "hi",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and wrong value", t, func() {
		properties := map[string]interface{}{
			testField: "sdflkjdfkljskldfj",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and one of correct values", t, func() {
		properties := map[string]interface{}{
			testField: testValues[1],
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})

	Convey("with the field present and one of correct values and different case", t, func() {
		properties := map[string]interface{}{
			testField: strings.ToUpper(testValues[1]),
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})
}

func TestCompareFilter(t *testing.T) {
	filter := CompareFilter("lte", testField, strconv.Itoa(testNumericValue))

	Convey("without the field present", t, func() {
		properties := map[string]interface{}{
			"foo": "hi",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and value too big", t, func() {
		properties := map[string]interface{}{
			testField: testNumericValue + 1,
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and value correct", t, func() {
		properties := map[string]interface{}{
			testField: testNumericValue,
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})

	Convey("with the field present and value correct and not numeric", t, func() {
		properties := map[string]interface{}{
			testField: strconv.Itoa(testNumericValue - 1),
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})
}

func TestNotFilter(t *testing.T) {
	innerFilter := TermFilter(testField, testValues[0])
	filter := NotFilter(innerFilter)

	Convey("without the field present", t, func() {
		properties := map[string]interface{}{
			"foo": "hi",
		}

		innerValid := innerFilter.Valid(properties)
		valid := filter.Valid(properties)
		So(innerValid, ShouldBeFalse)
		So(valid, ShouldBeTrue)
	})

	Convey("with the field present and wrong value", t, func() {
		properties := map[string]interface{}{
			testField: "sdflkjdfkljskldfj",
		}

		innerValid := innerFilter.Valid(properties)
		valid := filter.Valid(properties)
		So(innerValid, ShouldBeFalse)
		So(valid, ShouldBeTrue)
	})

	Convey("with the field present and one of correct values", t, func() {
		properties := map[string]interface{}{
			testField: testValues[0],
		}

		innerValid := innerFilter.Valid(properties)
		valid := filter.Valid(properties)
		So(innerValid, ShouldBeTrue)
		So(valid, ShouldBeFalse)
	})
}

func TestOrFilter(t *testing.T) {
	innerFilters := []Filter{TermFilter(testField, testValues[0]), TermFilter(testField, testValues[1])}
	filter := OrFilter(innerFilters...)

	Convey("without the field present", t, func() {
		properties := map[string]interface{}{
			"foo": "hi",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and wrong value", t, func() {
		properties := map[string]interface{}{
			testField: "sdflkjdfkljskldfj",
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeFalse)
	})

	Convey("with the field present and one of correct values", t, func() {
		properties := map[string]interface{}{
			testField: testValues[1],
		}

		valid := filter.Valid(properties)
		So(valid, ShouldBeTrue)
	})
}

func TestBroadcastPlatformFilter(t *testing.T) {
	Convey("live", t, func() {
		filter := BroadcastPlatformFilter("live")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})

		Convey("spectre broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "spectre",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("watch_party broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})

		Convey("premiere broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party_premiere",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})

		Convey("rerun broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party_rerun",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})
	})

	Convey("live_video", t, func() {
		filter := BroadcastPlatformFilter("live_video")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})

		Convey("spectre broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "spectre",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("watch_party broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("premiere broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party_premiere",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("rerun broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party_rerun",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})
	})

	Convey("playlist", t, func() {
		filter := BroadcastPlatformFilter("playlist")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("spectre broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "spectre",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})
	})

	Convey("premiere", t, func() {
		filter := BroadcastPlatformFilter("premiere")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("premiere broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party_premiere",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})
	})

	Convey("rerun", t, func() {
		filter := BroadcastPlatformFilter("rerun")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("rerun broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "watch_party_rerun",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})
	})

	Convey("mobile", t, func() {
		filter := BroadcastPlatformFilter("mobile")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("ios broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "ios",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})

		Convey("android broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "android",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})
	})

	Convey("ps4", t, func() {
		filter := BroadcastPlatformFilter("ps4")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("octodad broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "octodad",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})
	})

	Convey("xbox", t, func() {
		filter := BroadcastPlatformFilter("xbox")

		Convey("random broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "blah",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeFalse)
		})

		Convey("candybox broadcaster", func() {
			properties := map[string]interface{}{
				"usher.broadcaster": "candybox",
			}

			valid := filter.Valid(properties)
			So(valid, ShouldBeTrue)
		})
	})
}
