package query

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/web/jax/common/language"
)

type esStreamsQuery struct {
	From    int                      `json:"from,omitempty"`
	Size    int                      `json:"size"`
	Sort    []map[string]interface{} `json:"sort,omitempty"`
	Fields  []string                 `json:"_source,omitempty"`
	Filters struct {
		And []Filter `json:"and,omitempty"`
	} `json:"filter,omitempty"`
}

type SearchQuery struct {
	Limit      int
	Offset     int
	SortFields []string
	Fields     []string
	Filters    []Filter
	Locale     string
}

func (S SearchQuery) ToQuery() []byte {
	q := &esStreamsQuery{}
	q.Size = S.Limit
	if q.Size == 0 {
		q.Size = 100
	}

	q.From = S.Offset
	q.Fields = S.Fields
	q.Sort = []map[string]interface{}{
		map[string]interface{}{
			"usher.channel_count": map[string]interface{}{
				"order": "desc",
			},
		},
		map[string]interface{}{
			"usher.id": map[string]interface{}{
				"order": "desc",
			},
		},
	}

	if S.Locale != "" && language.ValidLocales[S.Locale] {
		q.Sort = append([]map[string]interface{}{
			map[string]interface{}{
				fmt.Sprintf("usher.weighted_language_ccu.%v", S.Locale): map[string]interface{}{
					"order":   "desc",
					"missing": 0,
				},
			},
		}, q.Sort...)
	}

	// Make the sort fields higher priority than the defaults.
	for _, sortField := range S.SortFields {
		switch sortField {
		case "skill":
			q.Sort = append([]map[string]interface{}{
				map[string]interface{}{
					"csgo.skill": map[string]interface{}{
						"order": "desc",
					},
				},
			}, q.Sort...)
		case "summoner_rank":
			q.Sort = append([]map[string]interface{}{
				map[string]interface{}{
					"leagueoflegends.summoner_rank_numeric": map[string]interface{}{
						"order": "desc",
					},
				},
			}, q.Sort...)
		case "recency":
			q.Sort = append([]map[string]interface{}{
				map[string]interface{}{
					"usher.stream_up_timestamp": map[string]interface{}{
						"order": "desc",
					},
				},
			}, q.Sort...)
		case "random":
			return RandomSortSearchQuery(S.Limit, S.Offset, S.Fields, S.Filters)
		}
	}

	q.Filters.And = S.Filters

	b, err := json.Marshal(q)
	if err != nil {
		return []byte{}
	}
	return b
}
