// Package main executes an integration test run in Runscope
// against jax.
package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"sync"
	"time"

	runscope "github.com/nextrevision/go-runscope"
)

const (
	bucketKey    = "ji54ym1dcl3b"   // the key for "Search and Discovery" bucket
	maxWait      = 30 * time.Second // max amount of time to wait for any given test to complete
	runscopeTest = "f9909885-787b-432f-b9db-962b125c3f4f"

	// test statuses
	pass         = "pass"
	fail         = "fail"
	initializing = "init"
	working      = "working"
	queued       = "queued"
	timeout      = "timeout"
	erroredOut   = "runscope error"
)

// Result is the final result of a Runscope test suite run.
type Result struct {
	name   string
	url    string
	status string

	*runscope.Result
}

var accessToken string
var failures = []Result{}
var isVerbose bool

func init() {
	accessToken = os.Getenv("RUNSCOPE_TOKEN")

	flag.BoolVar(&isVerbose, "v", false, "enable for more versbose output")
	flag.Parse()
}

func main() {
	fmt.Println()
	log.Println("Starting Runscope integration tests...")
	start := time.Now()

	if accessToken == "" {
		log.Fatalf("%-3s %q must be set on the environment to run tests!",
			"💀", "RUNSCOPE_TOKEN")
	}
	client := runscope.NewClient(runscope.Options{Token: accessToken})

	log.Println("Fetching test suite bucket...")

	test, err := client.GetTest(bucketKey, runscopeTest)
	if err != nil {
		log.Fatalln("[ERROR] Unable to get test %v: %v", runscopeTest, err)
	}

	log.Printf("Got test %q in %s", test.Name, since(start))
	log.Println("Triggering test...")

	triggerStart := time.Now()
	result, err := client.Trigger(test.TriggerURL)
	if err != nil {
		log.Fatalln("[ERROR] Failed to trigger the test: %v", err)
	}

	log.Printf("Triggered %d test suites (%d started ok, %d failed to start). Took %s",
		result.RunsTotal, result.RunsStarted, result.RunsFailed, since(triggerStart))

	// Iterate through all the tests returned.
	wg := sync.WaitGroup{}
	n := len(result.Runs)
	results := make(chan Result, n)
	wg.Add(n)

	for _, r := range result.Runs {
		go func(run runscope.TestRun) {
			defer wg.Done()
			runStart := time.Now()

			// Space the requests to Runscope out across some interval
			// so that we don't run into any rate limiting issues.
			time.Sleep(randomDuration(5, 15, time.Second))

			checkInterval := 10 * time.Second
			ticker := time.NewTicker(checkInterval)

			for range ticker.C {
				res, err := client.GetResult(bucketKey, run.TestID, run.TestRunID)
				if err != nil {
					log.Printf("%-3s %-24s unable to get test result: %s", "🚧", run.TestName, err)

					if statusCode := getStatusCode(err); statusCode >= 500 {
						log.Printf("%-3s %-24s", "♻️", "it's okay... we'll try again")
						continue // Forgo further execution of this case, and keep checking test status.
					}

					// Otherwise, terminate further test checking because
					// something is probably wrong with Runscope.
					res = runscope.Result{Result: erroredOut}
				}

				result := Result{name: run.TestName, url: run.TestRunURL, Result: &res}

				switch status := res.Result; status {
				case pass, fail, erroredOut:
					result.status = status

				default:
					// This case is met if the test has not completed yet.
					elapsed := time.Since(runStart)

					// Print the status of long-running tests regardless of verbosity setting
					// so the user knows that the test runner has not hung.
					if isVerbose || elapsed > 40*time.Second {
						log.Printf("%-3s %-24s %15s %10s", "⏱", run.TestName, status+" ...", since(getTime(res.StartedAt)))
					}

					// If the test has taken too long to finish, mark it as timed out.
					if elapsed > maxWait {
						result.status = timeout
					}
				}

				// Terminate the ticker loop as soon as we have a result
				if result.status != "" {
					results <- result
					ticker.Stop()
					return
				}
			}
		}(r)
	}

	// Wait in a separate thread so we can print results as we get them.
	// Otherwise, the runner would appear to be unresponsive.
	go func() {
		wg.Wait()
		close(results)
	}()

	template := "%-3s %-24s %15s %10s"
	for result := range results {
		switch status := result.status; status {
		case pass:
			log.Printf(template, "✅", result.name, "passed in ...", elapsed(result.StartedAt, result.FinishedAt))
		case timeout:
			log.Printf(template, "⏰", result.name, "timed out !!!", since(getTime(result.StartedAt)))
			failures = append(failures, result)
		case erroredOut:
			log.Printf(template, "❓", result.name, "runscope error!")
			failures = append(failures, result)
		case fail:
			log.Printf(template, "❌", result.name, "failed in ...", elapsed(result.StartedAt, result.FinishedAt))
			// Iterate and print failed assertions
			for _, req := range result.Requests {
				if req.Result == pass || len(req.URL) == 0 {
					// We check len(req.URL) because the API returns some blank results.
					// Let's ignore the blank requests for now.
					continue
				}

				log.Printf("\t%-4s %s", req.Method, req.URL)

				for _, a := range req.Assertions {
					if a.Result == pass {
						continue
					}

					var prop string
					if len(a.Property) > 0 {
						prop = fmt.Sprintf(".%s", a.Property)
					}

					log.Printf("\t\t- %s%s %s %s (actual: %s)",
						a.Source, prop, a.Comparison, a.TargetValue, a.ActualValue)
				}
			}
			failures = append(failures, result)

		default:
			log.Printf("%-3s %-24s unknown test run result %q", "⁉️", result.name, status)
			log.Printf("%-3s See: %s", "‼️", result.url)
		}
	}

	log.Printf("%-3s %-40s %10s\n", "🏁", "Done! Total run time...", since(start))

	if len(failures) > 0 {
		log.Printf("%-3s There were %d failures:", "🔥", len(failures))
		for _, fail := range failures {
			log.Printf("%-3s %-24s %s", "🔥", fail.name, fail.url)
		}
		log.Fatalln()
	}

	log.Printf("%-3s All tests passed!", "🎉")
	fmt.Println()
}
