package main

import (
	"context"
	"flag"
	"fmt"
	"net/http"
	"strings"
	"sync"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	spadeclient "code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/common/stats"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

var confPath = "jax.json"

type SpadeClient interface {
	spadeclient.Client
}

func main() {
	flag.StringVar(&confPath, "conf", confPath, "Path to configuration file")
	flag.Parse()

	conf, err := config.New(confPath)
	log.Init(conf)
	if err != nil {
		log.Printf("Error loading config: %s", err.Error())
		conf, err = config.NewFromEnvironment()
		if err != nil {
			log.Fatalf(err.Error())
		}
		log.Init(conf)
	}

	stats := stats.InitStatsd(conf)

	spadeClient, err := spadeclient.NewClient(
		spadeclient.InitHTTPClient(&http.Client{}),
		spadeclient.InitMaxConcurrency(1000),
		spadeclient.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			statName := fmt.Sprintf("service.spade.%s.%d", name, httpStatus)
			stats.TimingDuration(statName, dur, 1.0)
		}),
	)
	if err != nil {
		log.Fatalf("failed to spin up spade snapshotter: %v", err)
	}

	reader := db.NewElasticSearchReader(conf, false)

	sendToSpade(reader, spadeClient, stats)
}

func sendToSpade(reader db.JaxReader, spade SpadeClient, stats statsd.Statter) {
	t := time.Now()
	log.Printf("sending snapshot at %v", t)

	var wg sync.WaitGroup

	streams := reader.Scan("live_channels", 500, query.TTLAliveFilter())
	total := 0
	for ch := range streams {
		total += len(ch)
		for _, stream := range ch {
			props := convertToSpade(db.FlattenProperties(stream.Properties))
			props["sender_timestamp"] = t.Unix()

			wg.Add(1)
			go func(props map[string]interface{}) {
				defer wg.Done()
				spade.TrackEvent(context.Background(), "jax_live_channel", props)
			}(props)
		}
	}
	wg.Wait()

	log.Printf("done one cycle, found %v streams, time taken: %vs", total, time.Since(t).Seconds())
	stats.Gauge("spade.snapshot", int64(total), 1.0)

	// Give time for stats to send :shrug:
	time.Sleep(5 * time.Second)
}

// convertToSpade converts properties keys to only having dashes
func convertToSpade(props map[string]interface{}) map[string]interface{} {
	newProps := map[string]interface{}{}
	for k, v := range props {
		key := strings.Replace(k, ".", "-", -1)
		newProps[key] = v
	}

	return newProps
}
