package main

import (
	"context"
	"testing"

	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db"
	dbmocks "code.justin.tv/web/jax/db/mocks"
	"code.justin.tv/web/jax/db/query"
	"code.justin.tv/web/jax/spade_snapshot/mocks"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/mock"

	. "github.com/smartystreets/goconvey/convey"
)

func TestSendToSpade(t *testing.T) {
	log.Init(nil)

	Convey("sends data to spade", t, func() {
		reader := &dbmocks.JaxReader{}
		spade := &mocks.SpadeClient{}
		stats := &statsd.NoopClient{}

		outChan := make(chan []db.ChannelResult, 2)
		outChan <- []db.ChannelResult{
			db.ChannelResult{
				Properties: map[string]interface{}{
					"foo": "bar",
				},
			},
		}
		outChan <- []db.ChannelResult{
			db.ChannelResult{
				Properties: map[string]interface{}{
					"foo": "baz",
				},
			},
		}
		close(outChan)

		reader.On("Scan", "live_channels", 500, []query.Filter{query.TTLAliveFilter()}).Return(outChan)

		spade.On("TrackEvent", context.Background(), "jax_live_channel", mock.MatchedBy(func(data map[string]interface{}) bool {
			return data["foo"] == "bar"
		})).Return(nil)
		spade.On("TrackEvent", context.Background(), "jax_live_channel", mock.MatchedBy(func(data map[string]interface{}) bool {
			return data["foo"] == "baz"
		})).Return(nil)

		sendToSpade(reader, spade, stats)

		mock.AssertExpectationsForObjects(t, reader.Mock, spade.Mock)
	})
}

func TestConvertToSpade(t *testing.T) {
	Convey("converts correctly", t, func() {
		res := convertToSpade(map[string]interface{}{
			"foo.bar":     "whatever",
			"foo.bar.baz": "what",
		})

		So(res, ShouldResemble, map[string]interface{}{
			"foo-bar":     "whatever",
			"foo-bar-baz": "what",
		})
	})
}
