package kinesis

import (
	"encoding/json"
	"fmt"
	"time"

	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/common/usher"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	awsKinesis "github.com/aws/aws-sdk-go/service/kinesis"
)

const (
	region = "us-west-2"
)

// Record is a format required by kinesis
type Record struct {
	Action   string            `json:"action"`
	Document Document          `json:"document"`
	Options  map[string]string `json:"options"`
}

// Document is part of a format required by kinesis
type Document struct {
	ID     string  `json:"id"`
	Type   string  `json:"type"`
	Fields []Field `json:"fields"`
}

// Field is part of a format required by kinesis
type Field struct {
	Name  string      `json:"name"`
	Value interface{} `json:"value"`
	Type  string      `json:"type"`
}

// Channel is the input container for all information gathered by JAX
type Channel struct {
	Game            string      `json:"rails.meta_game"`
	Login           string      `json:"rails.channel"`
	Name            string      `json:"rails.display_name"`
	Status          string      `json:"rails.title"`
	Language        string      `json:"rails.broadcaster_language"`
	DirectoryHidden bool        `json:"rails.directory_hidden"`
	ChannelID       json.Number `json:"rails.channel_id"`
	ChannelCount    json.Number `json:"usher.channel_count"`
	MaxHeight       json.Number `json:"usher.max_height"`
	AvcProfile      string      `json:"usher.avc_profile"`
	AvcLevel        string      `json:"usher.avc_level"`
	Broadcaster     string      `json:"usher.broadcaster"`
	Hls             bool        `json:"usher.hls"`
	Orbis           bool        `json:"playstation.sce_platform"`
}

// Client is a wrapper over the Kinesis client and other config options
type Client interface {
	PublishStreams(chs []Channel) (err error)
	GetRecordsChannel() (chan *awsKinesis.GetRecordsOutput, error)
}

type Kinesis interface {
	PutRecords(*awsKinesis.PutRecordsInput) (*awsKinesis.PutRecordsOutput, error)
	DescribeStream(*awsKinesis.DescribeStreamInput) (*awsKinesis.DescribeStreamOutput, error)
	GetShardIterator(*awsKinesis.GetShardIteratorInput) (*awsKinesis.GetShardIteratorOutput, error)
	GetRecords(*awsKinesis.GetRecordsInput) (*awsKinesis.GetRecordsOutput, error)
}

type client struct {
	KinesisClient Kinesis
	Index         string
	StreamName    string
}

// GetClient retrieves the kinesis client if one exists, otherwise create it
func GetClient(credentials *credentials.Credentials, index, streamName string) Client {
	client := new(client)
	sess := session.New(&aws.Config{
		Region:      aws.String(region),
		Credentials: credentials,
	})

	client.KinesisClient = awsKinesis.New(sess)
	client.Index = index
	client.StreamName = streamName

	return client
}

func (T *client) generateRecord(ch Channel) (rec Record) {
	fields := []Field{
		Field{Name: "game", Value: ch.Game, Type: "string"},
		Field{Name: "login", Value: ch.Login, Type: "string"},
		Field{Name: "name", Value: ch.Name, Type: "string"},
		Field{Name: "status", Value: ch.Status, Type: "string"},
		Field{Name: "language", Value: ch.Language, Type: "string"},
		Field{Name: "updated_on", Value: time.Now().Unix(), Type: "integer"},
		Field{Name: "channel_count", Value: ch.ChannelCount, Type: "integer"},
		Field{Name: "max_height", Value: ch.MaxHeight, Type: "integer"},
		Field{Name: "avc_profile", Value: ch.AvcProfile, Type: "enum"},
		Field{Name: "avc_level", Value: ch.AvcLevel, Type: "enum"},
		Field{Name: "stream_type", Value: usher.StreamTypeMapping(ch.Broadcaster), Type: "string"},
		Field{Name: "hls", Value: ch.Hls, Type: "enum"},
		Field{Name: "orbis", Value: ch.Orbis, Type: "enum"},
	}
	doc := Document{
		ID:     string(ch.ChannelID),
		Type:   T.Index,
		Fields: fields,
	}
	options := map[string]string{
		"excluded_search_services": "swiftype",
	}
	return Record{
		Action:   "update",
		Document: doc,
		Options:  options,
	}
}

func (T *client) putRecords(records map[string]Record) (err error) {
	var params []*awsKinesis.PutRecordsRequestEntry
	for id, r := range records {
		payload, err := json.Marshal(r)
		if err != nil {
			return err
		}
		// sharding purposes
		param := &awsKinesis.PutRecordsRequestEntry{
			Data:            payload,
			ExplicitHashKey: aws.String(id),
			PartitionKey:    aws.String(id),
		}
		params = append(params, param)
	}
	recordRequest := &awsKinesis.PutRecordsInput{
		Records:    params,
		StreamName: aws.String(T.StreamName),
	}
	_, err = T.KinesisClient.PutRecords(recordRequest)
	if err != nil {
		fmt.Errorf("error: cannot put record into kinesis (%+v)", err)
	}
	return
}

// PublishStreams pushes streams to algolia through kinesis
func (T *client) PublishStreams(chs []Channel) (err error) {
	records := make(map[string]Record)
	for _, ch := range chs {
		rec := T.generateRecord(ch)
		records[string(ch.ChannelID)] = rec
	}
	return T.putRecords(records)
}

func (T *client) GetRecordsChannel() (chan *awsKinesis.GetRecordsOutput, error) {
	shards := []*awsKinesis.Shard{}
	hasMoreShards := true
	params := &awsKinesis.DescribeStreamInput{
		StreamName: aws.String(T.StreamName),
	}
	for {
		describeOutput, err := T.KinesisClient.DescribeStream(params)
		if err != nil {
			return nil, fmt.Errorf("failed to get shards %v", err)
		}

		shards = append(shards, describeOutput.StreamDescription.Shards...)
		hasMoreShards = *describeOutput.StreamDescription.HasMoreShards
		if hasMoreShards {
			lastShard := shards[len(shards)-1]
			params.ExclusiveStartShardId = lastShard.ShardId
		} else {
			break
		}
	}

	out := make(chan *awsKinesis.GetRecordsOutput, len(shards))

	for _, shard := range shards {
		params := &awsKinesis.GetShardIteratorInput{
			ShardId:           shard.ShardId,
			StreamName:        aws.String(T.StreamName),
			ShardIteratorType: aws.String("LATEST"),
		}

		output, err := T.KinesisClient.GetShardIterator(params)
		if err != nil {
			return nil, fmt.Errorf("failed to get shard iterator for %v %v", shard.ShardId, err)
		}

		shardIterator := output.ShardIterator
		go func(shardIterator *string) {
			for {
				recordRequest := &awsKinesis.GetRecordsInput{
					Limit:         aws.Int64(5000),
					ShardIterator: shardIterator,
				}
				recordsOutput, err := T.KinesisClient.GetRecords(recordRequest)
				if err != nil {
					log.Reportf("error getting records: %v", err)
				} else {
					out <- recordsOutput
					shardIterator = recordsOutput.NextShardIterator
					if shardIterator == nil {
						return
					}
				}

				time.Sleep(500 * time.Millisecond)
			}
		}(shardIterator)
	}

	return out, nil
}
