package liveness

import (
	"encoding/json"
	"strings"
	"time"

	"code.justin.tv/web/jax/common/language"
	"code.justin.tv/web/jax/common/log"
)

const languageCCUMultiplier = 50

var languageCCUUpdateInterval = 1 * time.Minute

func (T *LiveUpdater) updateChannelsLanguageCCU(ccuByLanguage map[string]map[string]int) {
	log.Printf("updating %v channels", len(ccuByLanguage))
	for channel, data := range ccuByLanguage {
		totalViewers := 0
		for _, ccu := range data {
			totalViewers += ccu
		}

		weightedData := map[string]int{}
		for locale, ccu := range data {
			weightedData[locale] = ccu*(languageCCUMultiplier-1) + totalViewers
		}

		T.writer.Update(channel, map[string]interface{}{
			"usher": map[string]interface{}{
				"weighted_language_ccu": weightedData,
			},
		})
	}
}

func (T *LiveUpdater) processLanguageCCU() error {
	ccuByLanguage := map[string]map[string]int{}

	type localeCCU struct {
		Channel string `json:"channel"`
		CCU     int    `json:"ccu"`
		Locale  string `json:"locale"`
	}

	ticker := time.NewTicker(languageCCUUpdateInterval)

	records, err := T.languageCCUStream.GetRecordsChannel()
	if err != nil {
		log.Reportf("failed to get language ccu records: %v\n", err)
		return err
	}

	for {
		select {
		case <-ticker.C:
			T.updateChannelsLanguageCCU(ccuByLanguage)
		case r := <-records:
			for _, record := range r.Records {
				var data localeCCU
				err := json.Unmarshal(record.Data, &data)
				if err != nil {
					log.Printf("malformed data %v: %v", string(record.Data), err)
					continue
				}

				if data.Channel == "" {
					continue
				}

				locale := strings.ToLower(data.Locale)

				if _, ok := language.ValidLocales[locale]; !ok {
					continue
				}

				if _, ok := ccuByLanguage[data.Channel]; !ok {
					ccuByLanguage[data.Channel] = map[string]int{}
				}

				ccuByLanguage[data.Channel][locale] = data.CCU
			}
		}
	}
}
