package liveness

import (
	"fmt"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/service/kinesis"

	"code.justin.tv/web/jax/common/log"
	dbmocks "code.justin.tv/web/jax/db/mocks"
	"code.justin.tv/web/jax/updater/kinesis/mocks"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestUpdateChannelsLanguageCCU(t *testing.T) {
	writer := &dbmocks.JaxWriter{}
	log.Init(nil)

	updater := &LiveUpdater{
		writer: writer,
	}

	Convey("works", t, func() {
		viewersA := 1
		viewersB := 2

		writer.On("Update", "testchannel", map[string]interface{}{
			"usher": map[string]interface{}{
				"weighted_language_ccu": map[string]int{
					"a": viewersA*languageCCUMultiplier + viewersB,
					"b": viewersB*languageCCUMultiplier + viewersA,
				},
			},
		}).Return(nil)

		updater.updateChannelsLanguageCCU(map[string]map[string]int{
			"testchannel": map[string]int{
				"a": 1,
				"b": 2,
			},
		})

		mock.AssertExpectationsForObjects(t, writer.Mock)
	})
}

func TestProcessLanguageCCU(t *testing.T) {
	writer := &dbmocks.JaxWriter{}
	kinesisClient := &mocks.Client{}
	log.Init(nil)

	updater := &LiveUpdater{
		writer:            writer,
		languageCCUStream: kinesisClient,
	}

	languageCCUUpdateInterval = 1 * time.Second

	Convey("errors", t, func() {
		err := fmt.Errorf("what")
		kinesisClient.On("GetRecordsChannel").Return(nil, err).Once()

		So(updater.processLanguageCCU(), ShouldEqual, err)
	})

	Convey("processes", t, func() {
		ch := make(chan *kinesis.GetRecordsOutput, 2)

		ch <- &kinesis.GetRecordsOutput{
			Records: []*kinesis.Record{
				&kinesis.Record{
					Data: []byte(`{}`),
				},
				&kinesis.Record{
					Data: []byte(`{"channel": "ch","locale":"lksdfj"}`),
				},
				&kinesis.Record{
					Data: []byte(`{"channel": "ch","locale":"en-US","ccu":1}`),
				},
			},
		}

		ch <- &kinesis.GetRecordsOutput{
			Records: []*kinesis.Record{
				&kinesis.Record{
					Data: []byte(`{"channel": "ch","locale":"en-GB","ccu":2}`),
				},
			},
		}

		kinesisClient.On("GetRecordsChannel").Return(ch, nil)

		writer.On("Update", "ch", map[string]interface{}{
			"usher": map[string]interface{}{
				"weighted_language_ccu": map[string]int{
					"en-us": 1*languageCCUMultiplier + 2,
					"en-gb": 2*languageCCUMultiplier + 1,
				},
			},
		}).Return(nil)

		go updater.processLanguageCCU()

		time.Sleep(3 * time.Second)

		mock.AssertExpectationsForObjects(t, writer.Mock, kinesisClient.Mock)
	})
}
