// Package liveness provides functions for adding and removing streams from Jax.
package liveness

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
	"code.justin.tv/web/jax/updater/kinesis"
	"github.com/cactus/go-statsd-client/statsd"
)

const (
	streamListScheme = "http"
	streamListPath   = "/channel_properties/live_hls_channels.json"

	updateInterval = time.Duration(15 * time.Second)
)

// LiveUpdater coordinates processes to update/add channels and remove expired channels.
type LiveUpdater struct {
	reader db.JaxReader
	writer db.JaxWriter

	usherHost     string
	lastUsherList []string

	languageCCUStream kinesis.Client

	stats statsd.Statter
}

// New creates a new LiveUpdater.
func New(reader db.JaxReader, writer db.JaxWriter, languageCCUStream kinesis.Client, usherHost string, stats statsd.Statter) *LiveUpdater {
	return &LiveUpdater{
		reader:            reader,
		writer:            writer,
		usherHost:         usherHost,
		languageCCUStream: languageCCUStream,
		stats:             stats,
	}
}

// Start begins the updating and reaping processes.
func (T *LiveUpdater) Start() {
	expiredSearchQuery := query.SearchQuery{
		Limit:   3000,
		Fields:  []string{"streamlist.last_updated"},
		Filters: []query.Filter{query.TTLExpiredFilter()},
	}

	go func() {
		for {
			err := T.updateLiveChannels(T.usherHost, T.stats)
			if err != nil {
				log.Reportf("failed to update streamlist: %v\n", err)
				if T.lastUsherList != nil && len(T.lastUsherList) > 0 {
					T.updateChannels(T.lastUsherList)
				}
			}
			res, err := T.reader.Search(nil, expiredSearchQuery)
			if res != nil {
				T.stats.Gauge("delete", int64(len(res.Hits)), 1)
				for _, s := range res.Hits {
					T.writer.Delete(s.Channel)
				}
			}

			allQuery := query.SearchQuery{
				Limit:  1,
				Offset: 0,
				Filters: []query.Filter{
					query.ExistsFieldFilter("usher.id"),
					query.TermFilter("rails.directory_hidden", "false"),
					query.NotFilter(query.TermFilter("usher.broadcaster", "spectre")),
				},
			}

			resp, qErr := T.reader.Search(nil, allQuery)
			if qErr != nil {
				log.Reportf("failed to get stream count: %v\n", qErr)
			} else {
				T.stats.Gauge("streams_debug._total", int64(resp.Total), 1)
			}
			time.Sleep(updateInterval)
		}
	}()

	go T.processLanguageCCU()
}

func (T *LiveUpdater) updateLiveChannels(usherHost string, stats statsd.Statter) (err error) {
	var resp *http.Response

	u := &url.URL{
		Scheme: streamListScheme,
		Host:   usherHost,
		Path:   streamListPath,
	}

	if resp, err = http.Get(u.String()); err != nil {
		stats.Inc("streamlist.error", 1, 1)
		return
	}
	stats.Inc("streamlist."+strconv.Itoa(resp.StatusCode), 1, 1)

	var m []map[string][]string

	b, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		err = fmt.Errorf("could not read response body from usher: %v\n", err)
		return
	}
	err = json.Unmarshal(b, &m)
	if err != nil {
		err = fmt.Errorf("could not unmarshal channel names from usher: %v\n", err)
		return
	}
	if len(m) != 1 {
		err = fmt.Errorf("json was in an unexpected format for list of streams\n")
		return
	}

	if _, ok := m[0]["channels"]; !ok {
		err = fmt.Errorf("json did not contain the expected 'channels' field\n")
		return
	}

	stats.Gauge("streamlist", int64(len(m[0]["channels"])), 1)

	T.updateChannels(m[0]["channels"])
	T.lastUsherList = m[0]["channels"]

	return
}

func (T *LiveUpdater) updateChannels(channels []string) {
	for _, ch := range channels {
		T.writer.Update(ch, map[string]interface{}{
			"streamlist": map[string]interface{}{
				"last_updated": time.Now().UTC(),
			},
		})
	}
}
