// Main updater process, spins up all the different types of updaters.

package main

import (
	"flag"
	"fmt"
	"net/http"
	"os"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"
	"golang.org/x/net/context"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/gometrics"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/common/stats"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/updater/kinesis"
	"code.justin.tv/web/jax/updater/liveness"
	"code.justin.tv/web/jax/updater/updater"
)

var (
	addr     = ":6065"
	confPath = "jax.json"

	httpReadTimeout = 10 * time.Second
	// WriteTimeout should be >30s to allow cpu profile downloads
	httpWriteTimeout = 45 * time.Second

	minutesToLive = 5
)

func main() {
	flag.StringVar(&addr, "addr", addr, "Bind address")
	flag.StringVar(&confPath, "conf", confPath, "Path to configuration file")
	flag.Parse()

	defer func() {
		if p := recover(); p != nil {
			log.Reportf("panic in updater %v", p)
			os.Exit(2)
		}
	}()

	conf, err := config.New(confPath)
	log.Init(conf)
	if err != nil {
		log.Printf("Error loading config: %s", err.Error())
		conf, err = config.NewFromEnvironment()
		if err != nil {
			log.Fatalf(err.Error())
		}
		log.Init(conf)
	}

	log.Printf("Config: %+v", *conf)
	log.Printf("Listening on %s\n", addr)

	hostName, err := os.Hostname()
	if err != nil {
		hostName = "unknown_host"
	}

	stats := stats.InitStatsd(conf)

	gometrics.Monitor(stats, time.Second*5)

	hystrixHandler := hystrix.NewStreamHandler()
	hystrixHandler.Start()
	go http.ListenAndServe(":7001", hystrixHandler)

	c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr: conf.StatsHostPort,
		Prefix:     fmt.Sprintf("jax.%s.%s.hystrix", conf.Environment, hostName),
	})
	if err != nil {
		log.Fatalf("could not initialize statsd client for hystrix: %v", err)
	}

	metricCollector.Registry.Register(c.NewStatsdCollector)

	reader := db.NewElasticSearchReader(conf, false)
	writer := db.NewElasticSearchWriter(conf)

	writer.CreateIndexIfNotExists(conf.ESShards, conf.ESReplicas)

	streamName := conf.KinesisIndex
	index := conf.KinesisLiveChannel
	if index == "" || streamName == "" {
		log.Fatalf("kinesis configuration variables not set")
	}
	searchUpdateStream := kinesis.GetClient(config.SearchKinesisCredentials(conf), index, streamName)

	streamName = conf.LanguageCCUKinesisIndex
	if streamName == "" {
		log.Fatalf("language ccu kinesis configuration variables not set")
	}

	languageCCUStream := kinesis.GetClient(config.LanguageCCUKinesisCredentials(conf), conf.Environment, streamName)

	liveUpdater := liveness.New(reader, writer, languageCCUStream, conf.UsherHost, stats)
	liveUpdater.Start()

	updater := updater.New(reader, writer, searchUpdateStream, stats,
		updater.NewUsherUpdater(),
		updater.NewRailsUpdater(),
		updater.NewChannelsUpdater(),
		updater.NewPlaystationUpdater(),
		updater.NewXboxUpdater(reader),
		updater.NewCsgoUpdater(reader),
		updater.NewLoLUpdater(reader, conf),
		updater.NewLoLRankUpdater(reader, conf),
		updater.NewZumaUpdater(reader),
		updater.NewHSUpdater(reader, conf),
		updater.NewOWUpdater(reader, conf),
		updater.NewPUBGUpdater(reader, conf),
		updater.NewPartnershipsUpdater(conf),
	)

	updater.Init(conf)
	go updater.Do()

	go func() {
		time.Sleep(time.Duration(minutesToLive) * time.Minute)
		os.Exit(0)
	}()

	if err := ListenAndServe(); err != nil {
		log.Fatalf("Error while listening: %v\n", err)
	}
}

// ListenAndServe makes it possible for us to gather debug information about this process.
func ListenAndServe() error {
	mux := http.NewServeMux()
	mux.Handle("/debug/", http.DefaultServeMux)
	mux.HandleFunc("/health", func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		fmt.Fprintf(w, "%s", "OK")
	})

	tracectx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		log.Fatalf("Error creating chitin handler: %s", err)
	}
	h := chitin.Handler(mux, chitin.SetBaseContext(tracectx))

	srv := &http.Server{
		Handler:      h,
		Addr:         addr,
		ReadTimeout:  httpReadTimeout,
		WriteTimeout: httpWriteTimeout,
	}

	return srv.ListenAndServe()
}
