package updater

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"reflect"
	"sync"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/jsonable"
	"code.justin.tv/web/jax/common/log"
	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const (
	channelsScheme = "http"
	channelsPath   = "/channels"
)

// ChannelsResult is the top most key in the Pylon Channels API response.
type ChannelsResult struct {
	Results []ChannelsProperties `json:"results"`
}

// ChannelsProperties describes all the fields returned by the Pylon Channels API which
// should be exposed through Jax.
type ChannelsProperties struct {
	ID                  int64               `json:"id" internal:"channel_id"`
	Name                string              `json:"name" internal:"channel"`
	DirectoryHidden     jsonable.NullBool   `json:"directory_hidden"`
	BroadcasterLanguage jsonable.NullString `json:"broadcaster_language"`
	Status              jsonable.NullString `json:"status" internal:"title"`
	ChannelViewCount    jsonable.NullInt64  `json:"views_count" internal:"channel_view_count"`
	MetaGame            *string             `json:"game" internal:"meta_game"`
	GameID              jsonable.NullInt64  `json:"game_id"`
}

type channelsUpdater struct {
	Client                  *http.Client
	HostPort                string
	PropertiesInternalNames map[string]int
	lock                    sync.RWMutex
	Stats                   statsd.Statter
}

// NewChannelsUpdater creates an updater for channels service
func NewChannelsUpdater() Updater {
	return &channelsUpdater{}
}

func (T *channelsUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Client = createHTTPClient()
	T.HostPort = conf.ChannelsHost
	T.Stats = stats

	hystrix.ConfigureCommand("channels_service_get", hystrix.CommandConfig{
		Timeout:               9000,
		MaxConcurrentRequests: 400,
		ErrorPercentThreshold: 25,
	})

	m, err := creflect.PropNames(*new(ChannelsProperties))
	if err != nil {
		panic(err)
	}

	T.PropertiesInternalNames = m
}

func (T *channelsUpdater) SourceField() string {
	return "channels"
}

func (T *channelsUpdater) QueryFilters() []query.Filter {
	return []query.Filter{}
}

func (T *channelsUpdater) UpdateTime() time.Duration {
	return 60 * time.Second
}

func (T *channelsUpdater) BufferSize() int {
	return 30
}

func (T *channelsUpdater) Fetch(chs []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var buf []byte

	if buf, err = T.get(chs); err != nil {
		log.Printf("%v", err.Error())
		return
	}

	if out, err = T.convert(buf); err != nil {
		return
	}

	return
}

func (T *channelsUpdater) get(chs []db.ChannelResult) (buf []byte, err error) {
	var resp *http.Response

	v := url.Values{}

	for _, ch := range chs {
		v.Add("name", ch.Channel)
	}

	u := &url.URL{
		Scheme:   channelsScheme,
		Host:     T.HostPort,
		Path:     channelsPath,
		RawQuery: v.Encode(),
	}

	res := make(chan []byte, 1)
	errors := hystrix.Go("channels_service_get", func() error {
		var channelsResponse []byte

		if resp, err = T.Client.Get(u.String()); err != nil {
			return err
		}

		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			b, _ := ioutil.ReadAll(resp.Body)

			if resp.StatusCode == http.StatusNotFound {
				log.Printf("Channel not found: %s", string(b))
				return nil
			}

			err = fmt.Errorf("error: bad status from channels (%d): %s", resp.StatusCode, string(b))
			return err
		}

		if channelsResponse, err = ioutil.ReadAll(resp.Body); err != nil {
			err = fmt.Errorf("error: invalid body received from channels service: %v", err)
			return err
		}

		res <- channelsResponse

		return nil
	}, nil)

	select {
	case buf = <-res:
		return
	case err = <-errors:
		return
	}

	return
}

func (T *channelsUpdater) convert(buf []byte) (map[string]map[string]interface{}, error) {
	data := ChannelsResult{}

	if err := json.Unmarshal(buf, &data); err != nil {
		return nil, err
	}

	out := make(map[string]map[string]interface{})

	for _, ch := range data.Results {
		chV := reflect.ValueOf(ch)

		valuesMap, err := constructValuesMap(chV, T.PropertiesInternalNames)

		if err != nil {
			return nil, err
		}
		out[ch.Name] = map[string]interface{}{"rails": valuesMap}
	}
	return out, nil
}
