package updater

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"math"
	"net/http"
	"net/url"
	"reflect"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/db"

	"code.justin.tv/web/jax/common/config"
	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db/query"
)

const (
	steamMatchUserParam = "caster"
	steamMatchPathURL   = "ICSGOStreamSystem_730/GetMatchScoreboard/v1"
	steamUserPathURL    = "ICSGOPlayers_730/GetPlayerProfile/v1"
	steamIDUserParam    = "steamID"
	steamScheme         = "https"

	csgoGameName = "Counter-Strike: Global Offensive"
)

// CsgoProperties describes all the fields returned by the CSGO API which
// should be exposed through Jax.
type CsgoProperties struct {
	Login      string `json:"login" internal:"channel"`
	SteamID    string `json:"steam_id"`
	Map        string `json:"map"`
	MapImg     string `json:"map_img"`
	MapName    string `json:"map_name"`
	Skill      int    `json:"skill"`
	Spectators int    `json:"spectators"`
}

type csgoUpdater struct {
	PropertiesInternalNames map[string]int
	Reader                  db.JaxReader
	Stats                   statsd.Statter
	conf                    *config.Config
}

type railsCsgoProperties struct {
	Login   string `json:"login" internal:"channel"`
	SteamID string `json:"steam_id"`
}

type steamPlayerResponse struct {
	Player playerProperties `json:"result"`
}

type steamMatchResponse struct {
	Match matchProperties `json:"result"`
}

type playerProperties struct {
	Competitive   playerCompetitive   `json:"competitive"`
	Commendations playerCommendations `json:"commendations"`
}

type playerCompetitive struct {
	Wins  int `json:"wins"`
	Skill int `json:"skill"`
}

type playerCommendations struct {
	Friendly int `json:"friendly"`
	Teaching int `json:"teaching"`
	Leader   int `json:"leader"`
}

type matchProperties struct {
	Map        string `json:"map"`
	MapName    string `json:"map_name"`
	MapImg     string `json:"map_img"`
	Spectators int    `json:"spectators"`
}

type jaxCsgoResponse struct {
	Hits  []jaxCsgoSteam `json:"hits"`
	Total int            `json:"_total"`
}

type jaxCsgoSteam struct {
	Channel    string                 `json:"channel"`
	Properties map[string]interface{} `json:"properties"`
}

type jaxSteamProperties struct {
	Channel string `json:"rails.channel"`
	SteamID string `json:"rails.steam_id"`
	Map     string `json:"csgo.map"`
	MapName string `json:"csgo.map_name"`
	MapImg  string `json:"csgo.map_img"`
	Skill   int    `json:"csgo.skill"`
}

func NewCsgoUpdater(reader db.JaxReader) Updater {
	return &csgoUpdater{Reader: reader}
}

func (T *csgoUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.conf = conf
	hystrix.ConfigureCommand("csgo_get", hystrix.CommandConfig{
		Timeout:               6000,
		MaxConcurrentRequests: 500,
		ErrorPercentThreshold: 50,
	})

	T.Stats = stats

	m, err := creflect.PropNames(*new(CsgoProperties))
	if err != nil {
		fmt.Errorf("error: cannot instantiate csgo updated (%+v)", err)
	}
	T.PropertiesInternalNames = m
}

func (T *csgoUpdater) SourceField() string {
	return "csgo"
}

func (T *csgoUpdater) UpdateTime() time.Duration {
	return 60 * time.Second
}

func (T *csgoUpdater) BufferSize() int {
	return 60
}

func (T *csgoUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.StringTermsFilter("rails.meta_game", []string{csgoGameName}),
		query.NotFilter(query.TermFilter("rails.steam_id", nil)),
		query.ExistsFieldFilter("rails.steam_id"),
	}
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *csgoUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var data map[string]CsgoProperties
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if data, err = T.getRailsData(chs); err != nil {
		return
	}
	if err = T.getMetadata(data); err != nil {
		return
	}
	if err = T.logger(data); err != nil {
		return
	}
	if out, err = T.convert(data); err != nil {
		return
	}
	return
}

func (T *csgoUpdater) getRailsData(chs []string) (data map[string]CsgoProperties, err error) {
	if len(chs) == 0 {
		return
	}
	fields := []string{"rails.channel", "rails.steam_id"}
	resultSet, err := T.Reader.BulkGetByChannel(chs, fields, "", T.BufferSize(), 0)

	var response jaxCsgoResponse
	resultBytes, err := json.Marshal(resultSet)
	if err = json.Unmarshal(resultBytes, &response); err != nil {
		return
	}
	data = make(map[string]CsgoProperties)
	var jaxProps jaxSteamProperties
	for _, ch := range response.Hits {
		flatten, err := json.Marshal(db.FlattenProperties(ch.Properties))

		if err = json.Unmarshal(flatten, &jaxProps); err != nil {
			fmt.Errorf("error: cannot unmarshal JAX data (%+v)", err)
			continue
		}
		csgoProps := CsgoProperties{
			Login:   jaxProps.Channel,
			SteamID: jaxProps.SteamID,
		}
		data[ch.Channel] = csgoProps
	}
	return
}

// getMetadata pulls steam metadata from an api endpoint exposed by valve
func (T *csgoUpdater) getMetadata(data map[string]CsgoProperties) (err error) {
	var matchResp steamMatchResponse
	var playerResp steamPlayerResponse
	for ch := range data {
		err = T.fetchSteamMetadata(data[ch].SteamID, steamMatchUserParam, steamMatchPathURL, &matchResp)
		properties := data[ch]
		properties.Map = matchResp.Match.Map
		properties.MapName = matchResp.Match.MapName
		properties.MapImg = matchResp.Match.MapImg
		properties.Spectators = matchResp.Match.Spectators
		data[ch] = properties
		if err != nil {
			return
		}

		err = T.fetchSteamMetadata(data[ch].SteamID, steamIDUserParam, steamUserPathURL, &playerResp)
		properties = data[ch]
		properties.Skill = playerResp.Player.Competitive.Skill
		data[ch] = properties
		if err != nil {
			return
		}

	}
	return
}

func (T *csgoUpdater) fetchSteamMetadata(steamID string, userParam string, pathURL string, response interface{}) (err error) {
	v := url.Values{}
	v.Set("key", T.conf.SteamAPIKey)
	v.Set(userParam, steamID)
	u := &url.URL{
		Scheme:   steamScheme,
		Host:     T.conf.SteamHost,
		Path:     pathURL,
		RawQuery: v.Encode(),
	}
	resp, err := http.Get(u.String())
	if err != nil {
		fmt.Printf("error: bad GET request %v", err)
		return
	}

	if resp.StatusCode != http.StatusOK {
		b, _ := ioutil.ReadAll(resp.Body)
		err = fmt.Errorf("error: bad status from steam (%d): %s", resp.StatusCode, string(b))
		return
	}

	decoder := json.NewDecoder(resp.Body)

	if err = decoder.Decode(&response); err != nil {
		fmt.Errorf("error: cannot decode response %+v", err)
	}
	return
}

// convert takes the data received through get and formats it in a structured map
func (T *csgoUpdater) convert(data map[string]CsgoProperties) (out map[string]map[string]interface{}, err error) {
	out = make(map[string]map[string]interface{})

	for ch := range data {
		chV := reflect.ValueOf(data[ch])
		valuesMap, err := constructValuesMap(chV, T.PropertiesInternalNames)
		if err != nil {
			return nil, err
		}
		out[data[ch].Login] = map[string]interface{}{T.SourceField(): valuesMap}
	}

	return
}

// verifies the integrity of data by checking against rails
func (T *csgoUpdater) logger(data map[string]CsgoProperties) (err error) {
	channels := []string{}
	for ch := range data {
		channels = append(channels, ch)
	}

	fields := []string{"csgo.map", "csgo.map_name", "csgo.map_img", "csgo.skill"}

	resultSet, err := T.Reader.BulkGetByChannel(channels, fields, "", T.BufferSize(), 0,
		query.NotFilter(query.TermFilter("csgo.map", nil)),
		query.NotFilter(query.TermFilter("csgo.map_name", nil)),
		query.NotFilter(query.TermFilter("csgo.map_img", nil)))

	var response jaxCsgoResponse
	resultBytes, err := json.Marshal(resultSet)
	if err = json.Unmarshal(resultBytes, &response); err != nil {
		return
	}
	var jaxProp, railsProp CsgoProperties
	var csgoProps jaxSteamProperties
	for _, ch := range response.Hits {
		flatten, err := json.Marshal(db.FlattenProperties(ch.Properties))
		if err = json.Unmarshal(flatten, &csgoProps); err != nil {
			fmt.Errorf("error: cannot unmarshal JAX data (%+v)", err)
			continue
		}
		railsProp = CsgoProperties{
			Map:     csgoProps.Map,
			MapName: csgoProps.MapName,
			MapImg:  csgoProps.MapImg,
			Skill:   csgoProps.Skill,
		}
		jaxProp = data[ch.Channel]
		if !(jaxProp.Map == railsProp.Map &&
			jaxProp.MapName == railsProp.MapName &&
			jaxProp.MapImg == railsProp.MapImg &&
			math.Abs(float64(jaxProp.Skill-railsProp.Skill)) < 2) {
			T.Stats.Inc("updater."+T.SourceField()+".presence", int64(1), 1.0)

		}
	}
	return
}
