package updater

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"

	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/hs/backend"
	"code.justin.tv/web/jax/common/hs/logic"
	. "code.justin.tv/web/jax/common/hs/models"
	"code.justin.tv/web/jax/db"

	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db/query"
)

const (
	hsGameID         = 138585
	hsGameStatePlay  = "GAME_PLAY"
	hsSpadeEventName = "hearthstone_stream_in_game"
)

type hsSpadeEvent struct {
	UserID               string `json:"user_id"`
	Login                string `json:"login"`
	BroadcastID          int64  `json:"broadcast_id"`
	InGame               bool   `json:"in_game"`
	Viewers              int64  `json:"viewers"`
	BroadcasterHeroName  string `json:"broadcaster_hero_name,omitempty"`
	BroadcasterHeroClass string `json:"broadcaster_hero_class,omitempty"`
	GameMode             string `json:"game_mode,omitempty"`
}

type hsUpdater struct {
	PropertiesInternalNames map[string]int
	Reader                  db.JaxReader
	SpadeClient             spade.Client
	Stats                   statsd.Statter
	Logic                   logic.Logic
}

func NewHSUpdater(reader db.JaxReader, conf *config.Config) Updater {
	b, err := backend.New(conf)
	if err != nil {
		fmt.Printf("error: could not create backend\n")
	}
	l, err := logic.New(conf, b)
	if err != nil {
		fmt.Printf("error: could not create logic\n")
	}
	return &hsUpdater{
		Reader: reader,
		Logic:  l,
	}
}

func (T *hsUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Stats = stats

	m, err := creflect.PropNames(*new(HSProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m

	spadeClient, err := spade.NewClient(
		spade.InitHTTPClient(&http.Client{}),
		spade.InitMaxConcurrency(T.BufferSize()),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			statName := fmt.Sprintf("service.spade.%s.%d", name, httpStatus)
			T.Stats.TimingDuration(statName, dur, 1.0)
		}),
	)
	T.SpadeClient = spadeClient
}

func (T *hsUpdater) SourceField() string {
	return "hearthstone"
}

func (T *hsUpdater) UpdateTime() time.Duration {
	return 10 * time.Second
}

func (T *hsUpdater) BufferSize() int {
	return 60
}

func (T *hsUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.TermFilter("rails.game_id", hsGameID),
	}
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *hsUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var data map[string]HSProperties
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if data, err = T.getChannelData(chs); err != nil {
		return
	}
	if err = T.getDeepMetadata(data); err != nil {
		return
	}

	out = make(map[string]map[string]interface{})
	for ch := range data {
		out_data := map[string]interface{}{
			"broadcaster_hero_name":  data[ch].BroadcasterHeroName,
			"broadcaster_hero_class": data[ch].BroadcasterHeroClass,
			"opponent_hero_name":     data[ch].OpponentHeroName,
			"opponent_hero_class":    data[ch].OpponentHeroClass,
			"game_mode":              data[ch].HSGameMode,
		}
		out[data[ch].Login] = map[string]interface{}{T.SourceField(): out_data}
	}
	if err = T.sendSpadeTracking(data); err != nil {
		return
	}
	return
}

// getChannelData obtains all metadata from rails and usher to populate HSProperties
func (T *hsUpdater) getChannelData(chs []string) (data map[string]HSProperties, err error) {
	data = make(map[string]HSProperties)
	if len(chs) == 0 {
		return
	}

	fields := []string{"usher.channel_count", "usher.id", "rails.channel_id"}

	resultSet, err2 := T.Reader.BulkGetByChannel(chs, fields, "", T.BufferSize(), 0)
	if err2 != nil {
		return data, err2
	}
	for _, channelResult := range resultSet.Hits {
		prop := data[channelResult.Channel]
		if usherProps, ok := channelResult.Properties["usher"].(map[string]interface{}); ok {
			if num, ok := usherProps["id"].(json.Number); ok {
				prop.BroadcastID, _ = num.Int64()
			}
			if num, ok := usherProps["channel_count"].(json.Number); ok {
				prop.Viewers, _ = num.Int64()
			}
		}
		if railsProps, ok := channelResult.Properties["rails"].(map[string]interface{}); ok {
			if num, ok := railsProps["channel_id"].(json.Number); ok {
				prop.TwitchID = num.String()
			}
		}
		data[channelResult.Channel] = prop
	}
	return
}

// getDeepMetadata obtains deepmetadata from vca to populate HSProperties
func (T *hsUpdater) getDeepMetadata(data map[string]HSProperties) (err error) {
	for ch, prop := range data {
		hsDeepmetadata, _ := T.Logic.GetHSDeepMetadata(prop.TwitchID)
		prop.Login = ch
		prop.GameState = hsDeepmetadata.State

		heroInfo := hsDeepmetadata.Heroes
		for _, hero := range heroInfo {
			if hero.Label == "broadcaster_hero" {
				prop.BroadcasterHeroName = hero.HeroName
				prop.BroadcasterHeroClass = hero.HeroClass
			} else if hero.Label == "opponent_hero" {
				prop.OpponentHeroName = hero.HeroName
				prop.OpponentHeroClass = hero.HeroClass
			}
		}

		gameModeInfo := hsDeepmetadata.GameMode
		prop.HSGameMode = gameModeInfo.Mode

		data[ch] = prop
	}
	return
}

func (T *hsUpdater) trackEvent(s hsSpadeEvent) (err error) {
	err = T.SpadeClient.TrackEvent(context.Background(), hsSpadeEventName, s)
	if err != nil {
		log.Printf(err.Error())
		T.Stats.Inc("updater."+T.SourceField()+"."+hsSpadeEventName+".spade_event_error", int64(1), 1.0)
	}
	return
}

func (T *hsUpdater) sendSpadeTracking(data map[string]HSProperties) (err error) {
	for ch := range data {
		s := hsSpadeEvent{
			UserID:               data[ch].TwitchID,
			Login:                data[ch].Login,
			BroadcastID:          data[ch].BroadcastID,
			InGame:               data[ch].GameState == hsGameStatePlay,
			Viewers:              data[ch].Viewers,
			BroadcasterHeroName:  data[ch].BroadcasterHeroName,
			BroadcasterHeroClass: data[ch].BroadcasterHeroClass,
			GameMode:             data[ch].HSGameMode,
		}
		go T.trackEvent(s)
	}
	return
}
