package updater

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"reflect"
	"strconv"
	"time"

	"golang.org/x/net/context"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/lol/backend"
	"code.justin.tv/web/jax/common/lol/logic"
	. "code.justin.tv/web/jax/common/lol/models"
	"code.justin.tv/web/jax/db"

	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db/query"
)

const (
	lolGameName       = "League of Legends"
	lolSpadeEventName = "league_stream_in_game"
)

type lolSpadeEvent struct {
	UserID      string `json:"user_id"`
	Login       string `json:"login"`
	SummonerID  int64  `json:"summoner_id"`
	BroadcastID int64  `json:"broadcast_id"`
	InGame      bool   `json:"in_game"`
	Viewers     int64  `json:"viewers"`
	ChampionID  int    `json:"champion_id"`
	Rank        string `json:"rank"`
	Division    string `json:"division"`
}

type lolUpdater struct {
	PropertiesInternalNames map[string]int
	Reader                  db.JaxReader
	SpadeClient             spade.Client
	Stats                   statsd.Statter
	Logic                   logic.Logic
}

func NewLoLUpdater(reader db.JaxReader, conf *config.Config) Updater {
	b, err := backend.New(conf)
	if err != nil {
		fmt.Printf("error: could not create backend\n")
	}
	l, err := logic.New(conf, b)
	if err != nil {
		fmt.Printf("error: could not create logic\n")
	}
	return &lolUpdater{
		Reader: reader,
		Logic:  l,
	}
}

func (T *lolUpdater) Init(conf *config.Config, stats statsd.Statter) {
	hystrix.ConfigureCommand("lol_get", hystrix.CommandConfig{
		Timeout:               6000,
		MaxConcurrentRequests: 500,
		ErrorPercentThreshold: 50,
	})
	spadeClient, err := spade.NewClient(
		spade.InitHTTPClient(&http.Client{}),
		spade.InitMaxConcurrency(T.BufferSize()),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			statName := fmt.Sprintf("service.spade.%s.%d", name, httpStatus)
			T.Stats.TimingDuration(statName, dur, 1.0)
		}),
	)
	T.SpadeClient = spadeClient
	T.Stats = stats

	m, err := creflect.PropNames(*new(LoLProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m
}

func (T *lolUpdater) SourceField() string {
	return "leagueoflegends"
}

func (T *lolUpdater) UpdateTime() time.Duration {
	return 30 * time.Second
}

func (T *lolUpdater) BufferSize() int {
	return 60
}

func (T *lolUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.StringTermsFilter("rails.meta_game", []string{lolGameName}),
	}
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *lolUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	data := map[string]LoLProperties{}

	for _, channel := range channels {
		id := channel.GetID()
		if id <= 0 {
			continue
		}

		props := LoLProperties{
			TwitchID: strconv.FormatInt(id, 10),
		}

		if num, ok := channel.Properties["usher.id"].(json.Number); ok {
			props.BroadcastID, _ = num.Int64()
		}
		if num, ok := channel.Properties["usher.channel_count"].(json.Number); ok {
			props.Viewers, _ = num.Int64()
		}

		data[channel.Channel] = props
	}

	if err = T.getSummonerData(data); err != nil {
		return
	}
	if err = T.sendSpadeTracking(data); err != nil {
		return
	}
	if out, err = T.convert(data); err != nil {
		return
	}
	return
}

// getSummonerData pulls summoner metadata
func (T *lolUpdater) getSummonerData(data map[string]LoLProperties) error {
	for ch := range data {
		lolProps, err := T.Logic.GetLoLData(data[ch].TwitchID)
		if err == nil && lolProps.IsConnected {
			lolProps.BroadcastID = data[ch].BroadcastID
			lolProps.Viewers = data[ch].Viewers
			lolProps.Login = ch
			data[ch] = lolProps
		}
	}

	return nil
}

// convert takes the data received through get and formats it in a structured map
func (T *lolUpdater) convert(data map[string]LoLProperties) (out map[string]map[string]interface{}, err error) {
	out = make(map[string]map[string]interface{})

	for ch := range data {
		chV := reflect.ValueOf(data[ch])
		valuesMap, err := constructValuesMap(chV, T.PropertiesInternalNames)
		if err != nil {
			return nil, err
		}
		out[data[ch].Login] = map[string]interface{}{T.SourceField(): valuesMap}
	}

	return
}

func (T *lolUpdater) trackEvent(s lolSpadeEvent) (err error) {
	err = T.SpadeClient.TrackEvent(context.Background(), lolSpadeEventName, s)
	if err != nil {
		log.Printf(err.Error())
		T.Stats.Inc("updater."+T.SourceField()+"."+lolSpadeEventName+".spade_event_error", int64(1), 1.0)
	}
	return
}

func (T *lolUpdater) sendSpadeTracking(data map[string]LoLProperties) (err error) {
	for ch := range data {
		s := lolSpadeEvent{
			UserID:      data[ch].TwitchID,
			Login:       data[ch].Login,
			SummonerID:  data[ch].SummonerID,
			BroadcastID: data[ch].BroadcastID,
			InGame:      data[ch].ChampionID != 0,
			Viewers:     data[ch].Viewers,
			ChampionID:  data[ch].ChampionID,
		}
		go T.trackEvent(s)
	}
	return
}
