package updater

import (
	"fmt"
	"reflect"
	"strconv"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/lol/backend"
	"code.justin.tv/web/jax/common/lol/logic"
	. "code.justin.tv/web/jax/common/lol/models"
	"code.justin.tv/web/jax/db"

	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db/query"
)

type lolRankUpdater struct {
	PropertiesInternalNames map[string]int
	Reader                  db.JaxReader
	Stats                   statsd.Statter
	Logic                   logic.Logic
}

func NewLoLRankUpdater(reader db.JaxReader, conf *config.Config) Updater {
	b, err := backend.New(conf)
	if err != nil {
		fmt.Printf("error: could not create backend\n")
	}
	l, err := logic.New(conf, b)
	if err != nil {
		fmt.Printf("error: could not create logic\n")
	}
	return &lolRankUpdater{
		Reader: reader,
		Logic:  l,
	}
}

func (T *lolRankUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Stats = stats

	m, err := creflect.PropNames(*new(LoLRankProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m
}

func (T *lolRankUpdater) SourceField() string {
	return "leagueoflegends_rank"
}

func (T *lolRankUpdater) UpdateTime() time.Duration {
	return 30 * time.Minute
}

func (T *lolRankUpdater) BufferSize() int {
	return 60
}

func (T *lolRankUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.StringTermsFilter("rails.meta_game", []string{lolGameName}),
	}
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *lolRankUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	data := map[string]LoLRankProperties{}

	for _, channel := range channels {
		id := channel.GetID()
		if id <= 0 {
			continue
		}

		lolProps, err := T.Logic.GetLoLRank(strconv.FormatInt(id, 10))
		if err == nil {
			lolProps.Login = channel.Channel
			data[channel.Channel] = lolProps
		}
	}

	if out, err = T.convert(data); err != nil {
		return
	}
	return
}

// convert takes the data received through get and formats it in a structured map
func (T *lolRankUpdater) convert(data map[string]LoLRankProperties) (out map[string]map[string]interface{}, err error) {
	out = make(map[string]map[string]interface{})

	for ch := range data {
		chV := reflect.ValueOf(data[ch])
		valuesMap, err := constructValuesMap(chV, T.PropertiesInternalNames)
		if err != nil {
			return nil, err
		}
		out[data[ch].Login] = map[string]interface{}{"leagueoflegends": valuesMap}
	}

	return
}
