package updater

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"

	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/overwatch/backend"
	"code.justin.tv/web/jax/common/overwatch/logic"
	. "code.justin.tv/web/jax/common/overwatch/models"
	"code.justin.tv/web/jax/db"

	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db/query"
)

const (
	owGameID         = 488552
	owGameStatePlay  = "GAME_PLAY"
	owSpadeEventName = "overwatch_stream_in_game"
)

type owSpadeEvent struct {
	UserID               string `json:"user_id"`
	Login                string `json:"login"`
	BroadcastID          int64  `json:"broadcast_id"`
	InGame               bool   `json:"in_game"`
	Viewers              int64  `json:"viewers"`
	BroadcasterCharacter string `json:"broadcaster_character,omitempty"`
}

type owUpdater struct {
	PropertiesInternalNames map[string]int
	Reader                  db.JaxReader
	SpadeClient             spade.Client
	Stats                   statsd.Statter
	Logic                   logic.Logic
}

func NewOWUpdater(reader db.JaxReader, conf *config.Config) Updater {
	b, err := backend.New(conf)
	if err != nil {
		fmt.Printf("error: could not create backend\n")
	}
	l, err := logic.New(conf, b)
	if err != nil {
		fmt.Printf("error: could not create logic\n")
	}
	return &owUpdater{
		Reader: reader,
		Logic:  l,
	}
}

func (T *owUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Stats = stats

	m, err := creflect.PropNames(*new(OverwatchProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m

	spadeClient, err := spade.NewClient(
		spade.InitHTTPClient(&http.Client{}),
		spade.InitMaxConcurrency(T.BufferSize()),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			statName := fmt.Sprintf("service.spade.%s.%d", name, httpStatus)
			T.Stats.TimingDuration(statName, dur, 1.0)
		}),
	)
	T.SpadeClient = spadeClient
}

func (T *owUpdater) SourceField() string {
	return "overwatch"
}

func (T *owUpdater) UpdateTime() time.Duration {
	return 10 * time.Second
}

func (T *owUpdater) BufferSize() int {
	return 60
}

func (T *owUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.TermFilter("rails.game_id", owGameID),
	}
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *owUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var data map[string]OverwatchProperties
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if data, err = T.getChannelData(chs); err != nil {
		return
	}
	if err = T.getDeepMetadata(data); err != nil {
		return
	}

	out = make(map[string]map[string]interface{})
	for ch := range data {
		out_data := map[string]interface{}{
			"broadcaster_character":         data[ch].BroadcasterCharacter,
			"broadcaster_character_role":    data[ch].BroadcasterCharacterRole,
			"broadcaster_character_ability": data[ch].BroadcasterCharacterAbility,
		}
		out[data[ch].Login] = map[string]interface{}{T.SourceField(): out_data}
	}
	if err = T.sendSpadeTracking(data); err != nil {
		return
	}
	return
}

// getChannelData obtains all metadata from rails and usher to populate OverwatchProperties
func (T *owUpdater) getChannelData(chs []string) (data map[string]OverwatchProperties, err error) {
	data = make(map[string]OverwatchProperties)
	if len(chs) == 0 {
		return
	}

	fields := []string{"usher.channel_count", "usher.id", "rails.channel_id"}

	resultSet, err2 := T.Reader.BulkGetByChannel(chs, fields, "", T.BufferSize(), 0)
	if err2 != nil {
		return data, err2
	}
	for _, channelResult := range resultSet.Hits {
		prop := data[channelResult.Channel]
		if usherProps, ok := channelResult.Properties["usher"].(map[string]interface{}); ok {
			if num, ok := usherProps["id"].(json.Number); ok {
				prop.BroadcastID, _ = num.Int64()
			}
			if num, ok := usherProps["channel_count"].(json.Number); ok {
				prop.Viewers, _ = num.Int64()
			}
		}
		if railsProps, ok := channelResult.Properties["rails"].(map[string]interface{}); ok {
			if num, ok := railsProps["channel_id"].(json.Number); ok {
				prop.TwitchID = num.String()
			}
		}
		data[channelResult.Channel] = prop
	}
	return
}

// getDeepMetadata obtains deepmetadata from vca to populate getDeepMetadata
func (T *owUpdater) getDeepMetadata(data map[string]OverwatchProperties) (err error) {
	for ch, prop := range data {
		owDeepmetadata, _ := T.Logic.GetOverwatchDeepMetadata(prop.TwitchID)
		prop.Login = ch
		prop.GameState = owDeepmetadata.State

		var detection_time_sec time.Duration = 0

		characterInfo := owDeepmetadata.Characters
		for _, character := range characterInfo {
			if character.Label == "broadcaster_hero" {
				// Report only the hero with highest detection time
				if character.DetectionTime >= detection_time_sec {
					prop.BroadcasterCharacter = character.Name
					prop.BroadcasterCharacterRole = character.Role
					prop.BroadcasterCharacterAbility = character.Ability
					detection_time_sec = character.DetectionTime
				}
			}
		}
		data[ch] = prop
	}
	return
}

func (T *owUpdater) trackEvent(s owSpadeEvent) (err error) {
	err = T.SpadeClient.TrackEvent(context.Background(), owSpadeEventName, s)
	if err != nil {
		log.Printf(err.Error())
		T.Stats.Inc("updater."+T.SourceField()+"."+owSpadeEventName+".spade_event_error", int64(1), 1.0)
	}
	return
}

func (T *owUpdater) sendSpadeTracking(data map[string]OverwatchProperties) (err error) {
	for ch := range data {
		s := owSpadeEvent{
			UserID:               data[ch].TwitchID,
			Login:                data[ch].Login,
			BroadcastID:          data[ch].BroadcastID,
			InGame:               data[ch].GameState == owGameStatePlay,
			Viewers:              data[ch].Viewers,
			BroadcasterCharacter: data[ch].BroadcasterCharacter,
		}
		go T.trackEvent(s)
	}
	return
}
