package updater

import (
	"fmt"
	"log"

	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/partnerships/backend"
	"code.justin.tv/web/jax/db"

	"code.justin.tv/web/jax/db/query"
)

type partnershipsUpdater struct {
	Stats   statsd.Statter
	Backend backend.PartnershipsBackend
}

func NewPartnershipsUpdater(conf *config.Config) Updater {
	backend, err := backend.New(conf)
	if err != nil {
		log.Fatalf("error: could not create partnerships backend: %s\n", err)
	}
	return &partnershipsUpdater{
		Backend: backend,
	}
}

func (T *partnershipsUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Stats = stats
}

func (T *partnershipsUpdater) SourceField() string {
	return "partnerships"
}

func (T *partnershipsUpdater) UpdateTime() time.Duration {
	return time.Hour
}

func (T *partnershipsUpdater) BufferSize() int {
	return 60
}

func (T *partnershipsUpdater) QueryFilters() []query.Filter {
	return []query.Filter{query.ExistsFieldFilter("rails.channel_id")}
}

func (T *partnershipsUpdater) Fetch(channels []db.ChannelResult) (map[string]map[string]interface{}, error) {
	data := make(map[string]map[string]interface{})
	var err error
	for _, channel := range channels {
		id := int(channel.GetID())
		if id <= 0 {
			err = fmt.Errorf("no id available for %s", channel.Channel)
			continue
		}

		isPartner, _err := T.Backend.IsPartner(id)
		if _err != nil {
			err = _err
			continue
		}

		data[channel.Channel] = map[string]interface{}{
			T.SourceField(): map[string]interface{}{
				"partner_program": isPartner,
			},
		}
	}
	return data, err
}
