// Defines an Updater that retrieves playstation properties.
// It queries properties from api.internal (via videoshim).
package updater

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"reflect"
	"strings"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/jsonable"
	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const (
	playstationPath   = "/api/internal/user/properties"
	playstationScheme = "http"
)

// PlaystationProperties describes all the fields returned by the Playstation API which
// should be exposed through Jax.
type PlaystationProperties struct {
	Login string `json:"login" internal:"-"`

	OrbisPlatform                   jsonable.NullString `json:"sce_platform"`
	OrbisOnlineID                   jsonable.NullString `json:"sce_user_online_id"`
	OrbisUserNpID                   jsonable.NullString `json:"sce_user_np_id"`
	OrbisUserCountry                jsonable.NullString `json:"sce_user_country"`
	OrbisTitleAgeRating             jsonable.NullInt64  `json:"sce_title_age_rating"`
	OrbisTitleLanguage              jsonable.NullString `json:"sce_title_language"`
	OrbisTitleID                    jsonable.NullString `json:"sce_title_id"`
	OrbisTitleSessionID             jsonable.NullString `json:"sce_title_session_id"`
	OrbisTitleAttribute             jsonable.NullString `json:"sce_title_attribute"`
	OrbisTitleProductID             jsonable.NullString `json:"sce_title_product_id"`
	OrbisTitleName                  jsonable.NullString `json:"sce_title_name"`
	OrbisTitleShortName             jsonable.NullString `json:"sce_title_short_name"`
	OrbisTitleStoreURL              jsonable.NullString `json:"sce_title_store_url"`
	OrbisTitleGenre                 jsonable.NullString `json:"sce_title_genre"`
	OrbisTitleMetadata              jsonable.NullString `json:"sce_title_metadata"`
	OrbisTitlePreset                jsonable.NullBool   `json:"sce_title_preset"`
	OrbisTitlePresetTextDescription jsonable.NullString `json:"sce_title_preset_text_description"`
	OrbisTitlePresetText1           jsonable.NullString `json:"sce_title_preset_text_1"`
	OrbisTitlePresetText2           jsonable.NullString `json:"sce_title_preset_text_2"`
	OrbisTitlePresetText3           jsonable.NullString `json:"sce_title_preset_text_3"`
	OrbisTitlePresetText4           jsonable.NullString `json:"sce_title_preset_text_4"`
	OrbisTitlePresetText5           jsonable.NullString `json:"sce_title_preset_text_5"`
}

type playstationUpdater struct {
	Client                  *http.Client
	HostPort                string
	PropertiesEncodedString string
	PropertiesInternalNames map[string]int
	conf                    *config.Config
}

// NewPlaystationUpdater creates an updater for playstation
func NewPlaystationUpdater() Updater {
	return &playstationUpdater{}
}

func (T *playstationUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.conf = conf
	hystrix.ConfigureCommand("playstation_get", hystrix.CommandConfig{
		Timeout:               6000,
		MaxConcurrentRequests: 500,
		ErrorPercentThreshold: 50,
	})

	T.Client = createHTTPClient()
	T.HostPort = conf.RailsHost

	m, err := creflect.PropNames(*new(PlaystationProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m
	T.PropertiesEncodedString = getPropertiesEncodedString(*new(PlaystationProperties))
}

func (T *playstationUpdater) SourceField() string {
	return "playstation"
}

func (T *playstationUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.TermFilter("usher.broadcaster", "octodad"),
	}
}

func (T *playstationUpdater) UpdateTime() time.Duration {
	return 120 * time.Second
}

func (T *playstationUpdater) BufferSize() int {
	return 20
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *playstationUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var buf []byte
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if buf, err = T.get(chs); err != nil {
		return
	}

	if out, err = T.convert(buf); err != nil {
		return
	}

	return
}

// get requests data from rails for each of the supplied channel names and
// returns the result.
func (T *playstationUpdater) get(chs []string) (buf []byte, err error) {
	res := make(chan []byte, 1)
	errors := hystrix.Go("playstation_get", func() error {
		v := url.Values{}
		v.Set("logins", strings.Join(chs, ","))
		v.Set("properties", T.PropertiesEncodedString)
		u := &url.URL{
			Scheme:   playstationScheme,
			Host:     T.HostPort,
			Path:     playstationPath,
			RawQuery: v.Encode(),
		}

		req, err := http.NewRequest("GET", u.String(), nil)
		if err != nil {
			return err
		}

		// Rewrite host header for api.internal.twitch.tv
		req.Header.Del("Host")
		req.Header.Set("Host", "api.internal.twitch.tv")

		resp, err := T.Client.Do(req)
		if err != nil {
			return err
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			b, _ := ioutil.ReadAll(resp.Body)
			err = fmt.Errorf("error: bad status from playstation (%d): %s", resp.StatusCode, string(b))
			return err
		}

		response, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			err = fmt.Errorf("error: invalid body received from playstation: %v", err)
			return err
		}

		res <- response

		return nil
	}, nil)

	select {
	case buf = <-res:
		return
	case err = <-errors:
		return
	}

	return
}

// convert takes the data received through get and formats it in a structured
// map.
func (T *playstationUpdater) convert(buf []byte) (map[string]map[string]interface{}, error) {
	var data []PlaystationProperties
	if err := json.Unmarshal(buf, &data); err != nil {
		return nil, err
	}

	out := make(map[string]map[string]interface{})

	for _, ch := range data {
		chV := reflect.ValueOf(ch)
		valuesMap, err := constructValuesMap(chV, T.PropertiesInternalNames)
		if err != nil {
			return nil, err
		}
		out[ch.Login] = map[string]interface{}{"playstation": valuesMap}
	}
	return out, nil
}
