// Package updater defines an updater process, which listens to a broker for lists of channels to update,
// retrieves properties for the channels from a source, and sends the data to the local updater process.
package updater

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"math/rand"
	"net"
	"net/http"
	"net/url"
	"reflect"
	"strings"
	"sync"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/jsonable"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
	"code.justin.tv/web/jax/updater/kinesis"
)

const (
	handlerScheme   = "http"
	handlerPath     = "/update"
	numWorkers      = 20
	indexSampleRate = 0.25
)

// Updater defines a process that reads data from a source for updating Jax.
type Updater interface {
	Init(conf *config.Config, stats statsd.Statter)
	SourceField() string
	UpdateTime() time.Duration
	BufferSize() int
	QueryFilters() []query.Filter
	Fetch(channels []db.ChannelResult) (map[string]map[string]interface{}, error)
}

// UpdateProcess is the process which pulls data from a source and updates it
// in the Jax backend.
type UpdateProcess struct {
	Updaters []Updater

	Reader       db.JaxReader
	Writer       db.JaxWriter
	LeaseChannel chan []db.ChannelResult

	searchUpdateStream kinesis.Client

	Stats statsd.Statter
}

// New creates a new UpdateProcess and sets all the necessary values.
func New(reader db.JaxReader, writer db.JaxWriter, searchUpdateStream kinesis.Client, stats statsd.Statter, updaters ...Updater) *UpdateProcess {
	return &UpdateProcess{
		Updaters:           updaters,
		Reader:             reader,
		Writer:             writer,
		searchUpdateStream: searchUpdateStream,
		Stats:              stats,
	}
}

func updatedFieldForSource(source string) string {
	return "updated_" + source + ".last_updated"
}

// Init initializes the underlying updater, client, and the logins channel.
func (T *UpdateProcess) Init(conf *config.Config) {
	for _, u := range T.Updaters {
		u.Init(conf, T.Stats)
	}

	filters := []query.Filter{
		query.TTLAliveFilter(),
	}

	T.LeaseChannel = T.Reader.Lease("updater", 100, filters...)
}

// Do starts updates lists of logins.
func (T *UpdateProcess) Do() {
	for i := 0; i < numWorkers; i++ {
		go T.processChannels()
	}
}

// processChannels goes through the cycle of:
// - Obtain data from this data source for these channels.
// - Update the database by sending the data to the updater process.
func (T *UpdateProcess) processChannels() {
	for chs := range T.LeaseChannel {
		// Ideally we want all problems here to error instead of just log,
		// but there are so many all related to the same issue: core running out of local ports
		if len(chs) == 0 {
			return
		}

		channelsWithFlattenedProps := []db.ChannelResult{}
		for _, c := range chs {
			channelsWithFlattenedProps = append(channelsWithFlattenedProps, db.ChannelResult{
				Channel:    c.Channel,
				Properties: db.FlattenProperties(c.Properties),
			})
		}

		var wg sync.WaitGroup
		output := make(chan updaterResponse, len(T.Updaters))
		for _, u := range T.Updaters {
			filters := append(u.QueryFilters(), query.OrFilter(
				query.MissingFieldFilter(updatedFieldForSource(u.SourceField())),
				query.ExpiredFieldFilter(updatedFieldForSource(u.SourceField()), u.UpdateTime()),
			))

			channels := []db.ChannelResult{}
			for _, c := range channelsWithFlattenedProps {
				// Check if channel satifies the filters for this updater
				valid := true
				for _, f := range filters {
					if !f.Valid(c.Properties) {
						valid = false
						break
					}
				}

				// Remove the last_updated, only add back in later if new properties are present
				delete(c.Properties, updatedFieldForSource(u.SourceField()))

				if valid {
					channels = append(channels, db.ChannelResult{
						Channel:    c.Channel,
						Properties: db.FlattenProperties(c.Properties),
					})
				}
			}

			if len(channels) > 0 {
				wg.Add(1)
				go T.getChannelProperties(&wg, u, channels, output)
			}
		}

		wg.Wait()
		close(output)

		properties := map[string]map[string]interface{}{}
		// Use existing properties as a start
		for _, ch := range channelsWithFlattenedProps {
			properties[ch.Channel] = ch.Properties
		}
		for props := range output {
			for ch, p := range props.properties {
				if _, ok := properties[ch]; !ok {
					properties[ch] = map[string]interface{}{}
				}
				// Overwrite old properties with new ones
				for k, v := range p {
					properties[ch][k] = v
				}
			}
		}

		T.update(properties)
		// Lower indexing rate by sampling
		if rand.Float64() < indexSampleRate {
			kinesisChs := []kinesis.Channel{}
			for _, c := range chs {
				tmp, _ := json.Marshal(properties[c.Channel])
				kinesisCh := kinesis.Channel{
					Language: "en", // default to English
				}
				json.Unmarshal(tmp, &kinesisCh)
				_, orbisEnabled := properties[c.Channel]["playstation.sce_platform"]
				kinesisCh.Orbis = orbisEnabled
				if !kinesisCh.DirectoryHidden {
					kinesisChs = append(kinesisChs, kinesisCh)
				}
			}
			T.searchUpdateStream.PublishStreams(kinesisChs)
		}
	}
}

type updaterResponse struct {
	updaterSource string
	properties    map[string]map[string]interface{}
}

func (T *UpdateProcess) getChannelProperties(wg *sync.WaitGroup, u Updater, channels []db.ChannelResult, output chan updaterResponse) {
	defer wg.Done()

	res, err := u.Fetch(channels)
	if err != nil {
		// While there are many connection timeouts, only Report the errors
		// which are not errors generated by a httpClient.Get() call.
		// This is not the preferred way, but unavoidable if we want to have
		// clear, useful logs in Sentry.
		if matched := strings.Contains(err.Error(), "error: "); matched {
			log.Reportf("%v fetch problem: %v\n", u.SourceField(), err)
		} else {
			log.Printf("%v fetch problem: %v\n", u.SourceField(), err)
		}
		return
	}

	T.Stats.Inc("updater."+u.SourceField()+".channels", int64(len(res)), 0.1)
	T.Stats.Inc("updater."+u.SourceField(), 1.0, 0.1)

	for ch := range res {
		res[ch] = db.FlattenProperties(res[ch])
		res[ch][updatedFieldForSource(u.SourceField())] = time.Now().UTC()
	}

	output <- updaterResponse{updaterSource: u.SourceField(), properties: res}
}

// update sends the newly acquired data for a set of channels to the updater.
func (T *UpdateProcess) update(data map[string]map[string]interface{}) error {
	for ch, props := range data {
		err := T.Writer.Update(ch, db.JsonifyProperties(props))
		if err != nil {
			return err
		}
	}

	return nil
}

// Utility methods
func createHTTPClient() *http.Client {
	client := http.DefaultClient

	client.Transport = &http.Transport{
		Proxy: http.ProxyFromEnvironment,
		Dial: (&net.Dialer{
			Timeout:   30 * time.Second,
			KeepAlive: 0,
		}).Dial,
		TLSHandshakeTimeout: 10 * time.Second,
		DisableKeepAlives:   true,
	}

	return client
}

func getPropertiesEncodedString(properties interface{}) string {
	var encoded []string
	for _, prop := range getColumns(properties) {
		if prop == "login" || prop == "id" {
			continue
		}
		encoded = append(encoded, url.QueryEscape(prop))
	}
	return strings.Join(encoded, ",")
}

// getColumns gets a list of propertynames given a certain struct.
func getColumns(in interface{}) []string {
	var cols []string
	inT := reflect.ValueOf(in).Type()
	for i := 0; i < inT.NumField(); i++ {
		cols = append(cols, strings.SplitN(inT.Field(i).Tag.Get("json"), ",", 2)[0])
	}
	return cols
}

// constructValuesMap takes a value and creates a map of property => property value
func constructValuesMap(value reflect.Value, propertiesInternalNames map[string]int) (map[string]interface{}, error) {
	values := make(map[string]interface{})
	for name, i := range propertiesInternalNames {
		iValue := value.Field(i).Interface()

		b, err := json.Marshal(iValue)
		if err != nil {
			return nil, err
		}
		values[name] = jsonable.RawString(b)
	}
	return values, nil
}

// callService makes a http request to the given URL and unmarshals response into data
func callService(client *http.Client, url *url.URL, data interface{}) (err error) {
	var resp *http.Response
	var buf []byte

	func() {
		if resp, err = client.Get(url.String()); err != nil {
			return
		}
		defer resp.Body.Close()
		if resp.StatusCode != http.StatusOK {
			b, _ := ioutil.ReadAll(resp.Body)
			err = fmt.Errorf("error: bad status from %s (%d): %s", url.Host, resp.StatusCode, string(b))
			return
		}
		if buf, err = ioutil.ReadAll(resp.Body); err != nil {
			err = fmt.Errorf("error: invalid body received from %s: %v", url.Host, err)
			return
		}
	}()

	if len(buf) > 0 {
		err = json.Unmarshal(buf, &data)
	}
	return err
}
