package updater

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"

	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/pubg/backend"
	"code.justin.tv/web/jax/common/pubg/logic"
	. "code.justin.tv/web/jax/common/pubg/models"
	"code.justin.tv/web/jax/db"

	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db/query"
)

const (
	pubgGameID         = 493057
	pubgGameStatePlay  = "GAME_PLAY"
	pubgSpadeEventName = "pubg_stream_in_game"
)

type pubgSpadeEvent struct {
	UserID           string `json:"user_id"`
	Login            string `json:"login"`
	BroadcastID      int64  `json:"broadcast_id"`
	InGame           bool   `json:"in_game"`
	Viewers          int64  `json:"viewers"`
	PlayerAliveCount int    `json:"player_alive_count,omitempty"`
	GameMode         string `json:"game_mode,omitempty"`
}

type pubgUpdater struct {
	PropertiesInternalNames map[string]int
	Reader                  db.JaxReader
	SpadeClient             spade.Client
	Stats                   statsd.Statter
	Logic                   logic.Logic
}

func NewPUBGUpdater(reader db.JaxReader, conf *config.Config) Updater {
	b, err := backend.New(conf)
	if err != nil {
		fmt.Printf("error: could not create backend\n")
	}
	l, err := logic.New(conf, b)
	if err != nil {
		fmt.Printf("error: could not create logic\n")
	}
	return &pubgUpdater{
		Reader: reader,
		Logic:  l,
	}
}

func (T *pubgUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Stats = stats

	m, err := creflect.PropNames(*new(PUBGProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m

	spadeClient, err := spade.NewClient(
		spade.InitHTTPClient(&http.Client{}),
		spade.InitMaxConcurrency(T.BufferSize()),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			statName := fmt.Sprintf("service.spade.%s.%d", name, httpStatus)
			T.Stats.TimingDuration(statName, dur, 1.0)
		}),
	)
	T.SpadeClient = spadeClient
}

func (T *pubgUpdater) SourceField() string {
	return "pubg"
}

func (T *pubgUpdater) UpdateTime() time.Duration {
	return 10 * time.Second
}

func (T *pubgUpdater) BufferSize() int {
	return 60
}

func (T *pubgUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.TermFilter("rails.game_id", pubgGameID),
	}
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *pubgUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var data map[string]PUBGProperties
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if data, err = T.getChannelData(chs); err != nil {
		return
	}
	if err = T.getDeepMetadata(data); err != nil {
		return
	}

	out = make(map[string]map[string]interface{})
	for ch := range data {
		outData := map[string]interface{}{
			"player_alive_count": data[ch].PlayerAliveCount,
			"game_mode":          data[ch].PUBGGameMode,
		}
		out[data[ch].Login] = map[string]interface{}{T.SourceField(): outData}
	}
	if err = T.sendSpadeTracking(data); err != nil {
		return
	}
	return
}

// getChannelData obtains all metadata from rails and usher to populate PUBGProperties
func (T *pubgUpdater) getChannelData(chs []string) (map[string]PUBGProperties, error) {
	data := make(map[string]PUBGProperties)
	if len(chs) == 0 {
		return data, nil
	}

	fields := []string{"usher.channel_count", "usher.id", "rails.channel_id"}

	resultSet, err := T.Reader.BulkGetByChannel(chs, fields, "", T.BufferSize(), 0)
	if err != nil {
		return data, err
	}
	for _, channelResult := range resultSet.Hits {
		prop := data[channelResult.Channel]
		if usherProps, ok := channelResult.Properties["usher"].(map[string]interface{}); ok {
			if num, ok := usherProps["id"].(json.Number); ok {
				prop.BroadcastID, _ = num.Int64()
			}
			if num, ok := usherProps["channel_count"].(json.Number); ok {
				prop.Viewers, _ = num.Int64()
			}
		}
		if railsProps, ok := channelResult.Properties["rails"].(map[string]interface{}); ok {
			if num, ok := railsProps["channel_id"].(json.Number); ok {
				prop.TwitchID = num.String()
			}
		}
		data[channelResult.Channel] = prop
	}
	return data, nil
}

// getDeepMetadata obtains deepmetadata from vca to populate PUBGProperties
func (T *pubgUpdater) getDeepMetadata(data map[string]PUBGProperties) (err error) {
	for ch, prop := range data {
		pubgDeepmetadata, err := T.Logic.GetPUBGDeepMetadata(prop.TwitchID)
		if err != nil {
			continue
		}
		prop.Login = ch
		prop.GameState = pubgDeepmetadata.State

		var detectionTimeSec time.Duration = 0
		playerAliveCounts := pubgDeepmetadata.PlayerAliveCounts
		for _, playerAliveCount := range playerAliveCounts {
			if playerAliveCount.DetectionTime >= detectionTimeSec {
				prop.PlayerAliveCount = playerAliveCount.Count
			}
		}

		gameModeInfo := pubgDeepmetadata.GameMode
		prop.PUBGGameMode = gameModeInfo.Mode

		data[ch] = prop
	}
	return
}

func (T *pubgUpdater) trackEvent(s pubgSpadeEvent) (err error) {
	err = T.SpadeClient.TrackEvent(context.Background(), pubgSpadeEventName, s)
	if err != nil {
		log.Printf(err.Error())
		T.Stats.Inc("updater."+T.SourceField()+"."+pubgSpadeEventName+".spade_event_error", int64(1), 1.0)
	}
	return
}

func (T *pubgUpdater) sendSpadeTracking(data map[string]PUBGProperties) (err error) {
	for ch := range data {
		s := pubgSpadeEvent{
			UserID:           data[ch].TwitchID,
			Login:            data[ch].Login,
			BroadcastID:      data[ch].BroadcastID,
			InGame:           data[ch].GameState == pubgGameStatePlay,
			Viewers:          data[ch].Viewers,
			PlayerAliveCount: data[ch].PlayerAliveCount,
			GameMode:         data[ch].PUBGGameMode,
		}
		go T.trackEvent(s)
	}
	return
}
