package updater

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"reflect"
	"strings"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/jsonable"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"

	creflect "code.justin.tv/web/jax/common/reflect"
)

const (
	railsPath   = "/api/internal/user/properties"
	railsScheme = "http"
)

// RailsProperties describes all the fields returned by the Rails API which
// should be exposed through Jax.
type RailsProperties struct {
	Login string `json:"login" internal:"channel"`
	ID    int64  `json:"id" internal:"channel_id"`

	DisplayName jsonable.NullString `json:"display_name"`
	Category    jsonable.NullString `json:"category"`
	Language    jsonable.NullString `json:"language"`

	SteamID jsonable.NullString `json:"steam_id"`

	XboxGamertag jsonable.NullString `json:"xbox_gamertag"`
	XboxTitleID  jsonable.NullString `json:"xbox_title_id"`
	RemoteIP     jsonable.NullString `json:"remote_ip"`
}

type railsUpdater struct {
	Client                  *http.Client
	PropertiesEncodedString string
	PropertiesInternalNames map[string]int
	Stats                   statsd.Statter
	conf                    *config.Config
}

// NewRailsUpdater creates an updater for rails
func NewRailsUpdater() Updater {
	return &railsUpdater{}
}

func (T *railsUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.conf = conf
	hystrix.ConfigureCommand("rails_get", hystrix.CommandConfig{
		Timeout:               6000,
		MaxConcurrentRequests: 500,
		ErrorPercentThreshold: 50,
	})

	T.Client = createHTTPClient()
	T.Stats = stats

	m, err := creflect.PropNames(*new(RailsProperties))
	if err != nil {
		panic(err)
	}

	T.PropertiesInternalNames = m
	T.PropertiesEncodedString = getPropertiesEncodedString(*new(RailsProperties))
}

func (T *railsUpdater) SourceField() string {
	return "rails"
}

func (T *railsUpdater) QueryFilters() []query.Filter {
	return []query.Filter{}
}

func (T *railsUpdater) UpdateTime() time.Duration {
	return 120 * time.Second
}

func (T *railsUpdater) BufferSize() int {
	return 30
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *railsUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	var buf []byte
	if buf, err = T.get(chs); err != nil {
		return
	}

	if out, err = T.convert(buf); err != nil {
		return
	}

	return
}

// get requests data from rails for each of the supplied channel names and
// returns the result.
func (T *railsUpdater) get(chs []string) (buf []byte, err error) {
	res := make(chan []byte, 1)
	errors := hystrix.Go("rails_get", func() error {
		v := url.Values{}
		v.Set("logins", strings.Join(chs, ","))
		v.Set("properties", T.PropertiesEncodedString)
		u := &url.URL{
			Scheme:   railsScheme,
			Host:     T.conf.RailsHost,
			Path:     railsPath,
			RawQuery: v.Encode(),
		}

		req, err := http.NewRequest("GET", u.String(), nil)
		if err != nil {
			return err
		}

		// Rewrite host header for api.internal.twitch.tv
		req.Header.Del("Host")
		req.Header.Set("Host", "api.internal.twitch.tv")

		resp, err := T.Client.Do(req)
		if err != nil {
			return err
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			b, _ := ioutil.ReadAll(resp.Body)
			err = fmt.Errorf("error: bad status from rails (%d): %s", resp.StatusCode, string(b))
			return err
		}

		response, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			err = fmt.Errorf("error: invalid body received from rails: %v", err)
			return err
		}

		res <- response

		return nil
	}, nil)

	select {
	case buf = <-res:
		return
	case err = <-errors:
		return
	}

	return
}

// convert takes the data received through get and formats it in a structured
// map.
func (T *railsUpdater) convert(buf []byte) (map[string]map[string]interface{}, error) {
	var data []RailsProperties
	if err := json.Unmarshal(buf, &data); err != nil {
		return nil, err
	}

	out := make(map[string]map[string]interface{})

	for _, ch := range data {

		chV := reflect.ValueOf(ch)
		valuesMap, err := constructValuesMap(chV, T.PropertiesInternalNames)
		if err != nil {
			return nil, err
		}
		out[ch.Login] = map[string]interface{}{"rails": valuesMap}
	}
	return out, nil
}
