// Defines a Updater that retrieves properties from usher.
// Queries video-api for usher properties.
package updater

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"reflect"
	"strconv"
	"strings"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/jsonable"
	creflect "code.justin.tv/web/jax/common/reflect"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const (
	usherPath   = "/channel_properties/list.json"
	usherScheme = "http"
)

// UsherProperties is the struct of Usher property Jax needs to expose.
type UsherProperties struct {
	Login          string              `json:"channel"         internal:"-"`
	ID             int64               `json:"id"`
	Part           int64               `json:"broadcast_part"`
	VideoUpdatedOn jsonable.Time       `json:"updated_on"`
	StreamUpTime   jsonable.Time       `json:"up_time"`
	StreamName     string              `json:"name"`
	Broadcaster    string              `json:"broadcaster"`
	DelayLength    int64               `json:"delay_length"`
	SDKClientID    jsonable.NullString `json:"client_id"`
	Hls            bool                `json:"hls"`
	AvcProfile     string              `json:"minimum_profile" internal:"avc_profile"`
	AvcLevel       string              `json:"minimum_level"   internal:"avc_level"`
	MaxHeight      int64               `json:"minimum_height"  internal:"max_height"`

	UpTimeEpoch int64 `json:"stream_up_timestamp"`

	Viewers      int64 `json:"channel_count"`
	SiteViewers  int64 `json:"site_count" internal:"-"`
	EmbedViewers int64 `json:"embed_count" internal:"-"`

	VideoHeight   int64   `json:"video_height"`
	VideoWidth    int64   `json:"video_width"`
	VideoCodec    string  `json:"video_codec"`
	VideoBitrate  float64 `json:"video_bitrate"`
	Configuration string  `json:"configuration"`
	AverageFps    float64 `json:"average_fps"`
}

type usherUpdater struct {
	Client                  *http.Client
	PropertiesEncodedString string
	PropertiesInternalNames map[string]int
	Stats                   statsd.Statter
	conf                    *config.Config
}

// NewUsherUpdater creates an updater for usher
func NewUsherUpdater() Updater {
	return &usherUpdater{}
}

func (T *usherUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.conf = conf
	hystrix.ConfigureCommand("usher_get", hystrix.CommandConfig{
		Timeout:               7000,
		MaxConcurrentRequests: 500,
		ErrorPercentThreshold: 50,
	})

	T.Client = createHTTPClient()

	m, err := creflect.PropNames(*new(UsherProperties))
	if err != nil {
		panic(err)
	}
	T.PropertiesInternalNames = m
	T.PropertiesEncodedString = getPropertiesEncodedString(*new(UsherProperties))
	T.Stats = stats
}

func (T *usherUpdater) SourceField() string {
	return "usher"
}

func (T *usherUpdater) QueryFilters() []query.Filter {
	return []query.Filter{}
}

func (T *usherUpdater) UpdateTime() time.Duration {
	return 30 * time.Second
}

func (T *usherUpdater) BufferSize() int {
	return 30
}

// Fetch is a wrapper around the sequential execution of get followed by
// convert.
func (T *usherUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var buf []byte
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if buf, err = T.get(chs); err != nil {
		return
	}

	if out, err = T.convert(buf); err != nil {
		return
	}

	return
}

// get requests data from usher for each of the supplied channel names and
// returns the result.
func (T *usherUpdater) get(chs []string) (buf []byte, err error) {
	res := make(chan []byte, 1)
	errors := hystrix.Go("usher_get", func() error {
		queryParts := []string{
			url.QueryEscape("columns") + "=" + T.PropertiesEncodedString,
			url.QueryEscape("channel") + "=" + url.QueryEscape(strings.Join(chs, ",")),
		}
		u := &url.URL{
			Scheme:   usherScheme,
			Host:     T.conf.UsherHost,
			Path:     usherPath,
			RawQuery: strings.Join(queryParts, "&"),
		}

		resp, err := T.Client.Get(u.String())
		if err != nil {
			return err
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			b, _ := ioutil.ReadAll(resp.Body)
			err = fmt.Errorf("error: bad status code from usher (%d): %s", resp.StatusCode, string(b))
			T.Stats.Inc("updater.usher.error", 1, 1)
			return err
		}

		response, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			err = fmt.Errorf("error: invalid body received from usher: %v", err)
			T.Stats.Inc("updater.usher.error", 1, 1)
			return err
		}

		res <- response

		return nil
	}, nil)

	select {
	case buf = <-res:
		return
	case err = <-errors:
		return
	}

	return
}

// convert takes the result from get() and formats it in a per-channel organised
// map of properties.
func (T *usherUpdater) convert(buf []byte) (out map[string]map[string]interface{}, err error) {
	var data []UsherProperties
	if err = json.Unmarshal(buf, &data); err != nil {
		return
	}

	out = make(map[string]map[string]interface{})
	for _, ch := range data {
		ch.StreamName = "live_user_" + ch.Login
		m := make(map[string]interface{})
		chV := reflect.ValueOf(ch)
		for name, i := range T.PropertiesInternalNames {
			value := chV.Field(i).Interface()
			// some types need a bit of conversion before remarshaling
			switch typed := value.(type) {
			case jsonable.Time:
				value = time.Time(typed)
			}

			var b []byte
			if b, err = json.Marshal(value); err != nil {
				return
			}
			m[name] = jsonable.RawString(b)
		}
		streamUp := strconv.FormatInt(time.Time(ch.StreamUpTime).Unix(), 10)
		m["stream_up_timestamp"] = jsonable.RawString(streamUp)
		out[ch.Login] = map[string]interface{}{"usher": m}
	}

	return
}
