package updater

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/web/jax/common/jsonable"
	"code.justin.tv/web/jax/common/xbox"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"

	"reflect"

	"code.justin.tv/web/jax/common/config"
	creflect "code.justin.tv/web/jax/common/reflect"

	"code.justin.tv/common/spade-client-go/spade"
	"github.com/cactus/go-statsd-client/statsd"
)

const (
	discoveryPath   = "/game/list"
	discoveryScheme = "http"

	jaxScheme          = "http"
	jaxPath            = "/streams"
	defaultJaxHostport = "localhost:6062"

	heartbeatSpadeEventName    = "xbox_heartbeat_update"
	heartbeatSpadeEventWorkers = 500
)

// XboxProperties describes all the fields returned by the Xbox API which
// should be exposed through Jax.
type XboxProperties struct {
	Login           string              `json:"login" internal:"channel"`
	LiveStatus      bool                `json:"live"`
	GameID          jsonable.NullInt64  `json:"game_id"`
	XboxTitleID     string              `json:"xbox_title_id"`
	XboxGamerTag    jsonable.NullString `json:"xbox_gamertag"`
	ChannelID       int                 `json:"id" internal:"channel_id"`
	Viewers         int                 `json:"views"`
	StreamStart     int                 `json:"stream_up_timestamp"`
	LiveBroadcastID int
}

type railsXboxProperties struct {
	Login        string              `json:"login" internal:"channel"`
	GameID       jsonable.NullInt64  `json:"game_id"`
	XboxGamerTag jsonable.NullString `json:"xbox_gamertag"`
	ID           jsonable.NullString `json:"id"`
}

type discoveryProperties struct {
	ID         int                  `json:"id"`
	Properties discoveryXboxTitleID `json:"properties"`
}

type discoveryXboxTitleID struct {
	XboxTitleID string `json:"xbox_title_id"`
}

type jaxResponse struct {
	Hits  []jaxStream `json:"hits"`
	Total int         `json:"_total"`
}

type jaxStream struct {
	Channel    string        `json:"channel"`
	Properties jaxProperties `json:"properties"`
}

type jaxProperties struct {
	Usher usherProperties `json:"usher"`
}

type usherProperties struct {
	Viewers         int `json:"channel_count"`
	StreamStart     int `json:"stream_up_timestamp"`
	LiveBroadcastID int `json:"id"`
}

// to make xbox live status testable
type heartbeatRequester func(body []xbox.HeartbeatRequestBody, xstsToken string) (validChs map[string]bool, err error)

type xboxUpdater struct {
	Client                       *http.Client
	XboxHostPort                 string
	RailsPropertiesEncodedString string
	XboxPropertiesInternalNames  map[string]int
	Stats                        statsd.Statter
	Reader                       db.JaxReader
	SpadeClient                  spade.Client
	heartbeatRequester           heartbeatRequester
	conf                         *config.Config
}

type spadeEvent struct {
	Login           string `json:"login"`
	LiveBroadcastID int    `json:"livebroadcastid"`
}

// NewXboxUpdater creates an updater for xbox
func NewXboxUpdater(reader db.JaxReader) Updater {
	return &xboxUpdater{Reader: reader}
}

func (T *xboxUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.conf = conf
	T.Client = createHTTPClient()

	spadeClient, err := spade.NewClient(
		spade.InitHTTPClient(&http.Client{}),
		spade.InitMaxConcurrency(T.BufferSize()),
		spade.InitStatHook(func(name string, httpStatus int, dur time.Duration) {
			statName := fmt.Sprintf("service.spade.%s.%d", name, httpStatus)
			T.Stats.TimingDuration(statName, dur, 1.0)
		}),
	)
	T.SpadeClient = spadeClient
	T.Stats = stats

	m, err := creflect.PropNames(*new(XboxProperties))
	if err != nil {
		panic(err)
	}

	T.XboxPropertiesInternalNames = m
	T.heartbeatRequester = xbox.MakeHeartbeatRequest
	T.RailsPropertiesEncodedString = getPropertiesEncodedString(*new(railsXboxProperties))
}

func (T *xboxUpdater) SourceField() string {
	return "xbox_heartbeat"
}

func (T *xboxUpdater) QueryFilters() []query.Filter {
	return []query.Filter{
		query.ExistsFieldFilter("rails.xbox_gamertag"),
		query.ExistsFieldFilter("rails.game_id"),
		query.ExistsFieldFilter("usher.broadcaster"),
		query.NotFilter(query.StringTermsFilter("usher.broadcaster", []string{"octodad", "candybox"})),
	}
}

func (T *xboxUpdater) UpdateTime() time.Duration {
	return 300 * time.Second
}

func (T *xboxUpdater) BufferSize() int {
	return 60
}

// Fetch is a wrapper around the sequential execution of gets followed by
// convert.
func (T *xboxUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var data = make(map[string]XboxProperties)
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	if data, err = T.getRailsData(chs); err != nil {
		return
	}
	if err = T.filterXboxTitleIDs(data); err != nil {
		return
	}
	if err = T.getViews(data); err != nil {
		return
	}
	if err = T.updateXboxLive(data); err != nil {
		return
	}
	if err = T.sendSpadeTracking(data); err != nil {
		return
	}
	if out, err = T.convert(data); err != nil {
		return
	}
	return
}

// convert takes the transformed data and formats it in a structured map.
func (T *xboxUpdater) convert(data map[string]XboxProperties) (out map[string]map[string]interface{}, err error) {
	out = make(map[string]map[string]interface{})

	for ch := range data {
		chV := reflect.ValueOf(data[ch])
		valuesMap, err := constructValuesMap(chV, T.XboxPropertiesInternalNames)
		if err != nil {
			return nil, err
		}
		out[data[ch].Login] = map[string]interface{}{"xbox_heartbeat": valuesMap}
	}

	return
}

// get requests data from rails for each of the supplied channel names and
// returns the result.
func (T *xboxUpdater) getRailsData(chs []string) (out map[string]XboxProperties, err error) {
	v := url.Values{}
	v.Set("logins", strings.Join(chs, ","))
	v.Set("properties", T.RailsPropertiesEncodedString)
	u := &url.URL{
		Scheme:   railsScheme,
		Host:     T.conf.RailsHost,
		Path:     railsPath,
		RawQuery: v.Encode(),
	}

	var data []XboxProperties
	if err = callService(T.Client, u, &data); err != nil {
		return
	}

	out = make(map[string]XboxProperties)

	for _, ch := range data {
		out[ch.Login] = ch
	}
	return out, nil
}

// requests data from discovery for each game id, and removes any channels that are not playing
// a game with an associated xbox_title_id
func (T *xboxUpdater) filterXboxTitleIDs(out map[string]XboxProperties) (err error) {
	xboxTitles := make(map[string]string)

	// filter out any channels that don't have a gameID. this can happen if
	// the game changes between jax filter and call to rails.
	for ch := range out {
		gameID := out[ch].GameID
		if gameID.AsNullString().Valid {
			xboxTitles[gameID.AsNullString().String] = ""
		} else {
			delete(out, ch)
		}
	}

	gameIDs := make([]string, 0, len(xboxTitles))
	for k := range xboxTitles {
		gameIDs = append(gameIDs, k)
	}

	// ping discovery
	v := url.Values{}
	v.Set("id", strings.Join(gameIDs, ","))
	v.Set("limit", strconv.Itoa(T.BufferSize()))
	u := &url.URL{
		Scheme:   discoveryScheme,
		Host:     T.conf.DiscoveryHost,
		Path:     discoveryPath,
		RawQuery: v.Encode(),
	}

	var data []discoveryProperties
	if err = callService(T.Client, u, &data); err != nil {
		return
	}

	// restructure array of map ids into map
	for _, game := range data {
		titleID := game.Properties.XboxTitleID
		id := strconv.Itoa(game.ID)
		if titleID != "" {
			xboxTitles[id] = titleID
		}
	}

	// add xbox_title_id data, or remove channel if game does not have an xbox_title_id
	for ch := range out {
		gameIDStr := out[ch].GameID.AsNullString().String
		if val := xboxTitles[gameIDStr]; val != "" {
			channel := out[ch]
			channel.XboxTitleID = val
			out[ch] = channel
		} else {
			delete(out, ch)
		}
	}

	return
}

func (T *xboxUpdater) getViews(out map[string]XboxProperties) (err error) {
	// build elasticsearch fields to query into jax
	channels := []string{}
	for ch := range out {
		channels = append(channels, ch)
	}

	fields := []string{"usher.channel_count", "usher.stream_up_timestamp", "usher.id"}

	resultSet, err := T.Reader.BulkGetByChannel(channels, fields, "", T.BufferSize(), 0)

	var response jaxResponse
	resultBytes, err := json.Marshal(resultSet)
	if err = json.Unmarshal(resultBytes, &response); err != nil {
		return
	}

	// update viewer counts from jax response
	for _, ch := range response.Hits {
		b := out[ch.Channel]
		b.Viewers = ch.Properties.Usher.Viewers
		b.StreamStart = ch.Properties.Usher.StreamStart
		b.LiveBroadcastID = ch.Properties.Usher.LiveBroadcastID
		out[ch.Channel] = b
	}

	return
}

func (T *xboxUpdater) updateXboxLive(out map[string]XboxProperties) (err error) {
	token, err := xbox.GetXToken(T.conf.HeartbeatServiceToken)
	if err != nil {
		log.Printf("error getting x token")
	}

	// formats into heartbeat api request input
	body := []xbox.HeartbeatRequestBody{}
	for ch := range out {
		longTitleid, err := strconv.ParseInt(out[ch].XboxTitleID, 16, 0)
		if err != nil {
			log.Printf("Error decoding xbox titleid")
		}

		if out[ch].StreamStart != 0 {
			payload := xbox.HeartbeatRequestBody{
				Titleid:     strconv.Itoa(int(longTitleid)),
				Broadcastid: ch,
				Gamertag:    out[ch].XboxGamerTag.String,
				Viewers:     strconv.Itoa(out[ch].Viewers),
				Started:     out[ch].StreamStart,
			}
			body = append(body, payload)
		} else {
			delete(out, ch)
		}
	}

	if len(body) > 0 {
		T.Stats.Inc("updater."+T.SourceField()+".presence", int64(len(body)), 1.0)
		validChs, err := T.heartbeatRequester(body, token)
		if err != nil {
			log.Printf("error making heartbeat request: %v", err.Error())
		}
		// filters out invalid channels not streaming from an xbox
		for ch := range out {
			updateCh := out[ch]
			_, ok := validChs[ch]
			updateCh.LiveStatus = ok
			out[ch] = updateCh
		}
	}

	return
}

func (T *xboxUpdater) trackEvent(s spadeEvent) (err error) {
	err = T.SpadeClient.TrackEvent(context.Background(), heartbeatSpadeEventName, s)
	if err != nil {
		log.Printf(err.Error())
		T.Stats.Inc("updater."+T.SourceField()+"."+heartbeatSpadeEventName+".spade_event_error", int64(1), 1.0)
	}
	return
}

func (T *xboxUpdater) sendSpadeTracking(out map[string]XboxProperties) (err error) {
	for ch := range out {
		if out[ch].LiveBroadcastID != 0 {
			s := spadeEvent{
				Login:           ch,
				LiveBroadcastID: out[ch].LiveBroadcastID,
			}
			go T.trackEvent(s)
		}
	}

	return
}
