package updater

import (
	"reflect"
	"testing"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/stats"
	"code.justin.tv/web/jax/common/xbox"
)

type heartbeatCase struct {
	name     string
	input    map[string]XboxProperties
	expected map[string]XboxProperties
}

func testHeartbeatRequester(body []xbox.HeartbeatRequestBody, xstsToken string) (validChs map[string]bool, err error) {
	liveList := map[string]bool{
		"muttnik":         true,
		"majornelson":     true,
		"dragonglove":     true,
		"chirashii":       false,
		"bescuseme":       false,
		"thehumanfallacy": false,
	}
	validChs = make(map[string]bool)
	for _, ch := range body {
		channel := ch.Broadcastid
		if _, ok := liveList[channel]; ok && liveList[channel] {
			validChs[channel] = ok
		}
	}
	return
}

func TestUpdateXboxLive(t *testing.T) {
	conf := &config.Config{
		Environment:           "development",
		StatsHostPort:         "graphite.internal.justin.tv:8125",
		HeartbeatServiceToken: "322",
	}
	T := xboxUpdater{
		heartbeatRequester: testHeartbeatRequester,
		Stats:              stats.InitStatsd(conf),
		conf:               conf,
	}
	mockLiveTrue := XboxProperties{
		LiveStatus:  true,
		XboxTitleID: "123",
		StreamStart: 1,
	}
	mockLiveFalse := XboxProperties{
		LiveStatus:  false,
		XboxTitleID: "123",
		StreamStart: 1,
	}
	for _, test := range []heartbeatCase{
		heartbeatCase{
			name:     "Test 1: empty",
			input:    map[string]XboxProperties{},
			expected: map[string]XboxProperties{},
		},
		heartbeatCase{
			name:     "Test 2: single false to true",
			input:    map[string]XboxProperties{"muttnik": mockLiveFalse},
			expected: map[string]XboxProperties{"muttnik": mockLiveTrue},
		},
		heartbeatCase{
			name:     "Test 3: single true to true",
			input:    map[string]XboxProperties{"muttnik": mockLiveTrue},
			expected: map[string]XboxProperties{"muttnik": mockLiveTrue},
		},
		heartbeatCase{
			name:     "Test 4: single true to false",
			input:    map[string]XboxProperties{"chirashii": mockLiveTrue},
			expected: map[string]XboxProperties{"chirashii": mockLiveFalse},
		},
		heartbeatCase{
			name:     "Test 5: single false to false",
			input:    map[string]XboxProperties{"chirashii": mockLiveFalse},
			expected: map[string]XboxProperties{"chirashii": mockLiveFalse},
		},
		heartbeatCase{
			name: "Test 6: many mixed to mixed",
			input: map[string]XboxProperties{
				"muttnik":         mockLiveTrue,
				"majornelson":     mockLiveFalse,
				"dragonglove":     mockLiveFalse,
				"chirashii":       mockLiveTrue,
				"bescuseme":       mockLiveFalse,
				"thehumanfallacy": mockLiveTrue,
			},
			expected: map[string]XboxProperties{
				"muttnik":         mockLiveTrue,
				"majornelson":     mockLiveTrue,
				"dragonglove":     mockLiveTrue,
				"chirashii":       mockLiveFalse,
				"bescuseme":       mockLiveFalse,
				"thehumanfallacy": mockLiveFalse,
			},
		},
	} {
		T.updateXboxLive(test.input)
		if eq := reflect.DeepEqual(test.input, test.expected); !eq {
			t.Errorf("incorrect update for case '%s', got '%+v' but expected '%+v'", test.name, test.input, test.expected)
		}
	}
}
