package updater

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"math"
	"net/http"
	"net/url"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	zumaapi "code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const (
	zumaScheme = "http"
	zumaPath   = "/v1/channels/communities/bulk_get_by_login"
)

type zumaUpdater struct {
	Client   *http.Client
	HostPort string
	Stats    statsd.Statter
	reader   db.JaxReader
}

// NewZumaUpdater creates an updater for zuma
func NewZumaUpdater(reader db.JaxReader) Updater {
	return &zumaUpdater{reader: reader}
}

func (T *zumaUpdater) Init(conf *config.Config, stats statsd.Statter) {
	T.Client = createHTTPClient()
	T.HostPort = conf.ZumaHost
	T.Stats = stats

	hystrix.ConfigureCommand("zuma_get", hystrix.CommandConfig{
		Timeout:               9000,
		MaxConcurrentRequests: 400,
		ErrorPercentThreshold: 25,
	})
}

func (T *zumaUpdater) SourceField() string {
	return "zuma"
}

func (T *zumaUpdater) QueryFilters() []query.Filter {
	return []query.Filter{}
}

func (T *zumaUpdater) UpdateTime() time.Duration {
	return 60 * time.Second
}

func (T *zumaUpdater) BufferSize() int {
	return 30
}

func (T *zumaUpdater) Fetch(channels []db.ChannelResult) (out map[string]map[string]interface{}, err error) {
	var chs = make([]string, 0)
	for _, channel := range channels {
		chs = append(chs, channel.Channel)
	}
	var zumaResp zumaapi.BulkGetChannelCommunitiesResponse
	if zumaResp, err = T.get(chs); err != nil {
		log.Printf("%v", err.Error())
		return
	}

	var channelViewCounts map[string]int64
	if channelViewCounts, err = T.fetchViewCounts(chs); err != nil {
		log.Printf("%v", err.Error())
		return
	}

	if out, err = T.convert(zumaResp, channelViewCounts); err != nil {
		return
	}

	return
}

func (T *zumaUpdater) fetchViewCounts(chs []string) (map[string]int64, error) {
	// fetch the viewer counts of each channel
	fields := []string{"usher.channel_count"}
	channels, err := T.reader.BulkGetByChannel(chs, fields, "", T.BufferSize(), 0)
	if err != nil {
		return nil, err
	}

	resp := map[string]int64{}
	for _, hit := range channels.Hits {
		resp[hit.Channel] = getViewerCount(hit)
	}
	return resp, nil
}

func getViewerCount(T db.ChannelResult) int64 {
	if usherProps, ok := T.Properties["usher"].(map[string]interface{}); ok {
		if num, ok := usherProps["channel_count"].(json.Number); ok {
			if count, err := num.Int64(); err == nil {
				return count
			}
		}
	}

	return 0
}

func (T *zumaUpdater) get(chs []string) (zumaapi.BulkGetChannelCommunitiesResponse, error) {
	var resp *http.Response

	bodyBytes, err := json.Marshal(zumaapi.BulkGetChannelCommunitiesRequest{ChannelLogins: chs})
	if err != nil {
		return zumaapi.BulkGetChannelCommunitiesResponse{}, err
	}

	u := &url.URL{
		Scheme: zumaScheme,
		Host:   T.HostPort,
		Path:   zumaPath,
	}

	res := make(chan []byte, 1)
	errors := hystrix.Go("zuma_get", func() error {
		var zumaResponse []byte

		if resp, err = T.Client.Post(u.String(), "application/json", bytes.NewReader(bodyBytes)); err != nil {
			return err
		}

		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			b, _ := ioutil.ReadAll(resp.Body)
			return fmt.Errorf("error: bad status from zuma (%d): %s", resp.StatusCode, string(b))
		}

		if zumaResponse, err = ioutil.ReadAll(resp.Body); err != nil {
			return fmt.Errorf("error: invalid body received from zuma: %v", err)
		}

		res <- zumaResponse

		return nil
	}, nil)

	select {
	case buf := <-res:
		data := zumaapi.BulkGetChannelCommunitiesResponse{}
		if err := json.Unmarshal(buf, &data); err != nil {
			return zumaapi.BulkGetChannelCommunitiesResponse{}, err
		}
		return data, nil
	case err := <-errors:
		return zumaapi.BulkGetChannelCommunitiesResponse{}, err
	}
}

func (T *zumaUpdater) convert(zumaResp zumaapi.BulkGetChannelCommunitiesResponse, channelCounts map[string]int64) (map[string]map[string]interface{}, error) {
	out := make(map[string]map[string]interface{})

	for _, ch := range zumaResp.Results {
		communityID := ""
		if len(ch.CommunityIDs) > 0 {
			communityID = ch.CommunityIDs[0]
		}
		weightedViewCount := generateZumaMagicWeightedViewerCount(channelCounts[ch.ChannelLogin], len(ch.CommunityIDs))
		out[ch.ChannelLogin] = map[string]interface{}{
			"zuma": map[string]interface{}{
				"community_id":          communityID,
				"community_ids":         ch.CommunityIDs,
				"weighted_viewer_count": weightedViewCount,
			},
		}
	}

	return out, nil
}

// (n^0.9) * (0.9^(m-1)) = Channel Score
// 	Where:
// 		n = # viewers
// 		m = # Communities the channel is participating in
func generateZumaMagicWeightedViewerCount(viewers int64, communities int) float64 {
	if viewers <= 0 {
		return 0
	} else if communities <= 0 {
		return float64(viewers)
	}
	n := float64(viewers)
	m := float64(communities)
	return math.Pow(n, 0.9) * math.Pow(0.9, (m-1))
}
