package api

import (
	"net/http"
	"time"

	"code.justin.tv/common/config"
	"code.justin.tv/web/upload-service/api/worker"
	"code.justin.tv/web/upload-service/backend"
	"code.justin.tv/web/upload-service/files"
	"code.justin.tv/web/upload-service/middleware"
	"code.justin.tv/web/upload-service/rpc/server"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/heroku/rollbar"
	log "github.com/sirupsen/logrus"
	"goji.io"
	"goji.io/pat"
)

type Server struct {
	*goji.Mux

	W *worker.Worker
}

func NewServerWithWorker(httpServer *goji.Mux, wo *worker.Worker) (*Server, error) {
	s := &Server{
		httpServer,
		wo,
	}

	instanceID := s.W.Backend.EC2InstanceID()
	s.Use(middleware.StartTimeMiddleware)
	s.Use(middleware.StatsMiddleware(instanceID, s.W.Stats))
	s.Use(middleware.LoggingMiddleware())

	s.HandleFunc(pat.Get("/"), s.EverythingIsFine)
	s.HandleFunc(pat.Get("/health"), s.EverythingIsFine)
	s.HandleFunc(pat.Post("/worker"), s.W.SQSRouter)

	rpcServer := server.NewServer(s.W.Backend)
	rpcHandler := uploader.NewUploaderServer(&rpcServer, nil, nil)
	s.Handle(pat.Post(uploader.UploaderPathPrefix+"*"), rpcHandler)

	return s, nil
}

func NewServer(httpServer *goji.Mux, backEnd backend.Backender, stats statsd.Statter, transformer transformations.ImageTransformer) (*Server, error) {
	monitoringStatter, err := statsd.NewBufferedClient(config.StatsdHostPort(), "", time.Second, 0)
	if err != nil {
		log.WithError(err).Error("Could not create statsd statter for feature service monitoring")

		monitoringStatter = nil
	}
	monitoringRollbar := worker.NewRollbarWrapper(rollbar.NewAsync("", config.Environment(), "", "", ""))

	endpoint := "http://" + config.App() + "." + config.Environment() + ".us-west2.twitch.tv/"
	twirpClient := uploader.NewUploaderProtobufClient(endpoint, &http.Client{})

	wo := &worker.Worker{
		Backend:           backEnd,
		Stats:             stats,
		Transformer:       transformer,
		Files:             &files.RealFileOperator{},
		MonitoringStatter: monitoringStatter,
		MonitoringRollbar: monitoringRollbar,
		TwirpClient:       twirpClient,
	}

	return NewServerWithWorker(httpServer, wo)
}

func (s *Server) EverythingIsFine(w http.ResponseWriter, r *http.Request) {
	w.Write([]byte("OK"))
}
