package worker

import (
	"context"
	"fmt"
	"os"
	"time"

	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"

	"code.justin.tv/web/upload-service/transformations"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	PreValidationErrorMsg  = "Failed pre-validation"
	PostValidationErrorMsg = "Failed post-validation"
	TransformationErrorMsg = "Failed to transform"
)

func mergeFields(a log.Fields, b log.Fields) log.Fields {
	result := make(log.Fields, len(a)+len(b))

	for _, f := range []log.Fields{a, b} {
		for k, v := range f {
			result[k] = v
		}
	}

	return result
}

func (wo *Worker) processUpload(ctx context.Context, upload *models.Upload, bucket, key string) error {
	uploadID := upload.UploadId

	logFields := log.Fields{
		"upload": upload,
		"bucket": bucket,
		"key":    key,
	}
	log.WithFields(logFields).Debug("Start process upload")

	startTime := time.Now()
	fileSize, err := wo.Backend.FileSizeS3(bucket, key)
	wo.monitorTiming("FileSizeS3", startTime, err != nil, logFields)
	if err != nil {
		return errors.Wrap(err, "Error getting file size from S3")
	}
	if err := fileSizeValidation(upload.PreValidation.FileSizeLessThan).ValidateSize(fileSize); err != nil {
		return errors.Wrap(err, "File on S3 is too large")
	}
	upload.PreValidation.FileSizeLessThan = 0
	wo.monitorFileSize(fileSize)

	startTime = time.Now()
	tmpfile, err := wo.Backend.DownloadS3(bucket, key)
	wo.monitorTiming("DownloadS3", startTime, err != nil, logFields)
	if err != nil {
		return errors.Wrap(err, "Error downloading file from S3")
	}
	defer wo.Files.Remove(tmpfile.Name())

	startTime = time.Now()
	err = wo.validate(tmpfile, transformations.NewFileInfo(uploadID), upload.PreValidation)
	wo.monitorTiming("PreValidation", startTime, err != nil, logFields)
	if err != nil {
		return errors.Wrap(err, PreValidationErrorMsg)
	}

	tmpDir, err := wo.Files.TempDir("", uploadID)
	if err != nil {
		return errors.Wrap(err, "Error allocating tempdir")
	}
	defer wo.Files.RemoveAll(tmpDir)

	outputInfos := make([]uploader.OutputInfo, len(upload.Outputs))
	for i, output := range upload.Outputs {
		info := transformations.NewFileInfo(uploadID)
		outputName := fmt.Sprintf("output[%d]: %s", i, output.Name)

		var outFile = tmpfile
		if len(output.Transformations) > 0 {
			startTime = time.Now()
			outFile, err = wo.Transformer.RunTransformations(tmpfile.Name(), tmpDir, fmt.Sprintf("%d", i), info, output.Transformations)
			wo.monitorTiming("Transformation", startTime, err != nil, logFields)
			if err != nil {
				return errors.Wrapf(WithStatus(err, uploader.Status_TRANSFORMATION_FAILED), "%s %s", TransformationErrorMsg, outputName)
			}
		}

		startTime = time.Now()
		err = wo.validate(outFile, info, output.PostValidation)
		wo.monitorTiming("PostValidation", startTime, err != nil, logFields)
		if err != nil {
			return errors.Wrapf(err, "%s %s", PostValidationErrorMsg, outputName)
		}

		outputInfos[i] = buildOutputInfo(upload.OutputPrefix, output.Name, info)
		mimeType := info.MimeType()

		startTime = time.Now()
		err := wo.Backend.UploadS3(outFile, outputInfos[i].Path, mimeType, output.GrantFullControl, output.GrantRead)

		wo.monitorTiming("UploadS3", startTime, err != nil, mergeFields(logFields, log.Fields{
			"outputPath":       outputInfos[i].Path,
			"grantFullControl": output.GrantFullControl,
			"grantRead":        output.GrantRead,
		}))

		if outFile != tmpfile {
			if err := outFile.Close(); err != nil {
				log.WithError(err).Warn("Error closing temp transformation file")
			}
		}
		if err != nil {
			return errors.Wrapf(err, "Failed to upload to s3 %s", outputName)
		}
	}

	if err := tmpfile.Close(); err != nil {
		log.WithError(err).Warn("Error closing temp file")
	}

	// We added this check in an attempt to debug `context canceled` errors from the pubsub client. Those errors could
	// be caused by two things:
	// 1. the overall request we are processing has been canceled by the server
	// 2. the request to pubsub timed out
	// This check is designed to check for #1 so that it can be disambiguated from #2.
	if err := ctx.Err(); err != nil {
		return errors.Wrap(err, "Discovered context error after processing")
	}

	// SNS Callback for successful upload happens here instead of worker.ProcessUpload because it would be
	// inconvenient to return the outputInfos from here.
	startTime = time.Now()
	err = wo.Backend.NotifyCallbacks(ctx, upload, uploader.Status_POSTPROCESS_COMPLETE, outputInfos)
	wo.monitorTiming("PublishSNS", startTime, err != nil, mergeFields(logFields, log.Fields{
		"callback": upload.Callback,
		"status":   uploader.Status_POSTPROCESS_COMPLETE,
	}))
	return err
}

func (wo *Worker) validate(tmpfile *os.File, info *transformations.FileInfo, validation models.Validation) StatusError {
	// If multiple validations fail, only information about the first one is returned.
	imageValidations := buildImageValidations(validation, tmpfile)
	if len(imageValidations) > 0 {
		wo.Transformer.FillInfoFromFile(tmpfile, info)
	}

	for _, validator := range imageValidations {
		if err := validator.Validate(info); err != nil {
			return err
		}
	}

	return nil
}

func buildImageValidations(validation models.Validation, tmpfile *os.File) (result []Validator) {
	convertValidationDeprecatedFields(&validation)
	if validation.Format != "" {
		result = append(result, imageFormatValidation{Expected: validation.Format, File: tmpfile})
	}
	if validation.FileSizeLessThan != 0 {
		result = append(result, fileSizeValidation(validation.FileSizeLessThan))
	}
	for _, aspectRatioConstraint := range validation.AspectRatioConstraints {
		result = append(result, aspectRatioValidation(aspectRatioConstraint))
	}
	for _, widthConstraint := range validation.WidthConstraints {
		result = append(result, widthValidation(widthConstraint))
	}
	for _, heightConstraint := range validation.HeightConstraints {
		result = append(result, heightValidation(heightConstraint))
	}

	// If there was no isImage validation, add one at the beginning.
	if validation.Format == "" && len(result) > 0 {
		result = append([]Validator{imageFormatValidation{"image", tmpfile}}, result...)
	}

	return result
}

func convertValidationDeprecatedFields(v *models.Validation) {
	if v.AspectRatio != 0 {
		v.AspectRatioConstraints = append(v.AspectRatioConstraints, models.Constraint{
			Value: v.AspectRatio,
			Test:  "=",
		})
	}
	addValidationDimensionConstraint(v, v.MinimumSize, ">=")
	addValidationDimensionConstraint(v, v.MaximumSize, "<=")
}

func addValidationDimensionConstraint(v *models.Validation, dimension *models.Dimension, test string) {
	if dimension != nil {
		v.WidthConstraints = append(v.WidthConstraints, models.Constraint{
			Value: float64(dimension.Width),
			Test:  test,
		})
		v.HeightConstraints = append(v.HeightConstraints, models.Constraint{
			Value: float64(dimension.Height),
			Test:  test,
		})
	}
}

func buildOutputInfo(prefix, nameTemplate string, info *transformations.FileInfo) uploader.OutputInfo {
	outputName := info.ExecTemplate(nameTemplate)
	outputPath := prefix + outputName
	result := uploader.OutputInfo{
		Path:         outputPath,
		Name:         outputName,
		NameTemplate: nameTemplate,
	}
	if info.IsImage {
		result.Format = info.Format
		result.Width = uint32(info.Width)
		result.Height = uint32(info.Height)
		result.Dimensions = info.Params[transformations.DimParam]
	}
	if info.Size != 0 {
		result.FileSize = info.Size
	}
	return result
}
