package worker

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"

	"os"

	"code.justin.tv/web/upload-service/backend/mocks"
	fileMocks "code.justin.tv/web/upload-service/files/mocks"
	transformMocks "code.justin.tv/web/upload-service/transformations/mocks"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/suite"
)

type BackendTestSuite struct {
	suite.Suite
}

func (suite *BackendTestSuite) TestBuildOutputInfo() {
	info := transformations.NewFileInfo("upload-id")
	info.IsImage = true
	info.Size = int64(13016)
	info.SetWidth(195)
	info.SetHeight(209)
	info.SetFormat("jpg")

	prefix := "s3://bucket/path/"
	template := "asdf{{UploadID}}_{{UploadID}}__{{Dimensions}}{{Extension}}"
	finalName := "asdfupload-id_upload-id__195x209.jpg"

	expectedResult := uploader.OutputInfo{
		Path:         prefix + finalName,
		Name:         finalName,
		NameTemplate: template,
		Format:       "jpg",
		Width:        195,
		Height:       209,
		Dimensions:   "195x209",
		FileSize:     13016,
	}

	result := buildOutputInfo(prefix, template, info)
	suite.Require().Equal(expectedResult, result)
}

func (suite *BackendTestSuite) TestBuildValidations() {
	validation := models.Validation{
		Format:           "jpg",
		FileSizeLessThan: 14 << 10,
		AspectRatioConstraints: []models.Constraint{
			{195.0 / 209.0, "="},
		},
		WidthConstraints: []models.Constraint{
			{190, ">="},
			{200, "<="},
		},
		HeightConstraints: []models.Constraint{
			{205, ">="},
			{210, "<="},
		},
	}
	validators := buildImageValidations(validation, nil)
	suite.Require().Equal(7, len(validators))

	expected := []Validator{
		imageFormatValidation{"jpg", nil},
		fileSizeValidation(14 << 10),
		aspectRatioValidation(models.Constraint{195.0 / 209.0, "="}),
		widthValidation(models.Constraint{190, ">="}),
		widthValidation(models.Constraint{200, "<="}),
		heightValidation(models.Constraint{205, ">="}),
		heightValidation(models.Constraint{210, "<="}),
	}
	suite.matchExpected(validators, expected)
}

func (suite *BackendTestSuite) TestConvertValidations() {
	validation := models.Validation{
		Format:           "jpg",
		FileSizeLessThan: 14 << 10,
		AspectRatio:      195.0 / 209.0,
		MinimumSize: &models.Dimension{
			Height: 205,
			Width:  190,
		},
		MaximumSize: &models.Dimension{
			Height: 210,
			Width:  200,
		},
	}
	validators := buildImageValidations(validation, nil)
	suite.Require().Equal(7, len(validators))

	expected := []Validator{
		imageFormatValidation{"jpg", nil},
		fileSizeValidation(14 << 10),
		aspectRatioValidation(models.Constraint{195.0 / 209.0, "="}),
		widthValidation(models.Constraint{190, ">="}),
		widthValidation(models.Constraint{200, "<="}),
		heightValidation(models.Constraint{205, ">="}),
		heightValidation(models.Constraint{210, "<="}),
	}
	suite.matchExpected(validators, expected)
}

func (suite *BackendTestSuite) matchExpected(validators, expected []Validator) {
	for _, v := range validators {
		found := false
		var i int
		var e Validator
		for i, e = range expected {
			if e == v {
				found = true
				break
			}
		}
		suite.Require().True(found, fmt.Sprintf("Unexpected validator: %#+v", v))
		expected = append(expected[:i], expected[i+1:]...)
	}
}

func (suite *BackendTestSuite) TestContextCanceled() {
	mockBackend := &mocks.Backender{}
	mockFiles := &fileMocks.FileOperations{}
	mockTransformer := &transformMocks.ImageTransformer{}
	worker := Worker{
		Backend:     mockBackend,
		Stats:       &statsd.NoopClient{},
		Files:       mockFiles,
		Transformer: mockTransformer,
	}

	bucket := "bucket"
	key := "key"
	tmpfileName := "/path/to/file"
	upload := &models.Upload{
		UploadId: "upload-id",
	}

	mockBackend.On("FileSizeS3", bucket, key).Return(int64(1), nil)
	mockBackend.On("DownloadS3", bucket, key).Return(os.NewFile(10, tmpfileName), nil)
	mockFiles.On("Remove", tmpfileName).Return(nil)
	mockFiles.On("TempDir", "", upload.UploadId).Return("/path/to", nil)
	mockFiles.On("RemoveAll", "/path/to").Return(nil)

	// No setup for mockBackend.On("NotifyCallbacks"), we expect it not to be called.

	ctx, cancel := context.WithCancel(context.Background())
	cancel()

	err := worker.processUpload(ctx, upload, bucket, key)
	suite.Require().Error(err)
}

func TestBackend(t *testing.T) {
	suite.Run(t, new(BackendTestSuite))
}
