package worker

import (
	"fmt"

	"code.justin.tv/web/upload-service/rpc/uploader"
)

const MaxRetries int = 5

type RetryableError struct {
	AttemptNumber int
	TriesTotal    int
	UploadID      string
	Wraps         error
}

func (e *RetryableError) Error() string {
	return fmt.Sprintf("Retryable (%d/%d): %s", e.AttemptNumber, e.TriesTotal, e.Wraps.Error())
}

func (e *RetryableError) Cause() error {
	return e.Wraps
}

func (e *RetryableError) ShouldRetry() bool {
	// This is not >= 0 because AttemptNumber starts at 1.
	return e.AttemptNumber > 0 && e.AttemptNumber < e.TriesTotal
}

func (e *RetryableError) Status() uploader.Status {
	if e.ShouldRetry() {
		return uploader.Status_POSTPROCESS_RETRYING
	}
	return Status(e.Wraps)
}

type StatusError interface {
	error
	Status() uploader.Status
}

type withStatus struct {
	cause  error
	status uploader.Status
}

func (t *withStatus) Error() string {
	return t.cause.Error()
}

func (t *withStatus) Cause() error {
	return t.cause
}

func (t *withStatus) Status() uploader.Status {
	return t.status
}

func WithStatus(err error, status uploader.Status) StatusError {
	if err == nil {
		return nil
	}

	return &withStatus{err, status}
}

// Find the first cause of an error that is a StatusError.
func CauseWithStatus(err error) StatusError {
	if err == nil {
		return nil
	}

	type causer interface {
		Cause() error
	}

	for {
		if status, ok := err.(StatusError); ok {
			return status
		}

		if cause, ok := err.(causer); ok {
			err = cause.Cause()
		} else {
			return nil
		}
	}
}

// Get the status code from a (potentially wrapped) error.
// If there is no error, assume POSTPROCESS_COMPLETE. If no error has status information, assume PROCESSING_FAILED.
func Status(err error) uploader.Status {
	if err == nil {
		return uploader.Status_POSTPROCESS_COMPLETE
	}

	status := CauseWithStatus(err)
	if status != nil {
		return status.Status()
	}

	return uploader.Status_PROCESSING_FAILED
}
