package mocks

import (
	"sync"
)

// rollbarMock mocks an async rollbar client.
// Functionality based off of heroku/rollbar 0.4.2
// rollbarMock should not be instantiated directly, instead use NewRollbarMock
type rollbarMock struct {
	CurrentToken string
	BodyChannel  chan string
	WaitGroup    sync.WaitGroup

	NetworkDelay chan struct{}
	SignalStart  chan struct{}
	SignalEnd    chan struct{}

	ReconstructedMsgs []struct{ Token, Body string }
}

// NewRollbarMock mocks the behavior of heroku/rollbar.NewAsync
func NewRollbarMock() *rollbarMock {
	mock := &rollbarMock{
		BodyChannel:       make(chan string),
		NetworkDelay:      make(chan struct{}),
		SignalStart:       make(chan struct{}),
		SignalEnd:         make(chan struct{}),
		ReconstructedMsgs: []struct{ Token, Body string }{},
	}
	go func() {
		for body := range mock.BodyChannel {
			mock.post(body)
			mock.WaitGroup.Done()
		}
	}()
	return mock
}

func (r *rollbarMock) Wait() {
	r.SignalStart <- struct{}{}
	r.WaitGroup.Wait()
}
func (r *rollbarMock) SetToken(token string) {
	r.CurrentToken = token
}
func (r *rollbarMock) SetCustom(custom map[string]interface{}) {}

func (r *rollbarMock) ErrorWithExtras(level string, err error, extras map[string]interface{}) {
	r.WaitGroup.Add(1)
	r.BodyChannel <- err.Error()
}

// rollbarMock.post mocks the behavior of heroku/rollbar.AsyncClient.post
func (r *rollbarMock) post(body string) {
	r.ReconstructedMsgs = append(r.ReconstructedMsgs, struct{ Token, Body string }{r.CurrentToken, body})
	<-r.NetworkDelay
	r.SignalEnd <- struct{}{}
}
