package worker

import (
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/web/upload-service/models"

	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	log "github.com/sirupsen/logrus"
)

type monitoringResult struct {
	uploadID  string
	startTime time.Time
	retErr    error
}

func (wo *Worker) processUploadMonitoring(monitoring models.Monitoring, result monitoringResult) {
	if monitoring.GrafanaPrefix != "" {
		wo.processUploadGrafana(monitoring.GrafanaPrefix, result)
	}
	if result.retErr != nil {
		if monitoring.SNSTopic != "" {
			wo.processUploadSNS(monitoring.SNSTopic, result)
		}
		if monitoring.RollbarToken != "" {
			wo.processUploadRollbar(monitoring.RollbarToken, result)
		}
	}
}

func (wo *Worker) processUploadSNS(snsTopic string, result monitoringResult) {
	logEntry := log.WithFields(log.Fields{
		"uploadID": result.uploadID,
		"snsTopic": snsTopic,
		"retErr":   result.retErr,
	})

	message := &uploader.StatusResponse{
		UploadId: result.uploadID,
		Status:   Status(result.retErr),
		Message:  result.retErr.Error(),
	}

	messageJSON, err := json.Marshal(message)
	if err != nil {
		logEntry.WithError(err).Warn("Error marshalling json while publishing to feature service SNS")
	}

	publishInput := sns.PublishInput{
		Message:  aws.String(string(messageJSON)),
		TopicArn: aws.String(snsTopic),
	}
	_, err = wo.Backend.SNS().Publish(&publishInput)
	if err != nil {
		logEntry.WithError(err).Warn("Error publishing to feature service SNS")
	}
}

func (wo *Worker) processUploadGrafana(grafanaPrefix string, result monitoringResult) {
	duration := time.Since(result.startTime)
	processResult := "success"
	if result.retErr != nil {
		if strings.Contains(result.retErr.Error(), PreValidationErrorMsg) {
			processResult = "errorPreValidation"
		} else if strings.Contains(result.retErr.Error(), TransformationErrorMsg) {
			processResult = "errorTransformation"
		} else if strings.Contains(result.retErr.Error(), PostValidationErrorMsg) {
			processResult = "errorPostValidation"
		}
	}
	err := wo.MonitoringStatter.TimingDuration(fmt.Sprintf("%s.%s", grafanaPrefix, processResult), duration, 1)
	if err != nil {
		log.WithFields(log.Fields{
			"uploadID":      result.uploadID,
			"grafanaPrefix": grafanaPrefix,
			"retErr":        result.retErr,
		}).WithError(err).Warn("Error publishing to feature service statsd")
	}
}

func (wo *Worker) processUploadRollbar(rollbarToken string, result monitoringResult) {
	wo.MonitoringRollbar.QueueRollbar(rollbarToken, result.retErr, result.uploadID)
}

func (wo *Worker) monitorTiming(operation string, startTime time.Time, hasErr bool, logFields log.Fields) {
	duration := time.Since(startTime)

	result := "success"
	if hasErr {
		result = "error"
	}

	logEntry := log.WithFields(logFields).WithFields(log.Fields{
		"operation": operation,
		"duration":  duration,
		"result":    result,
	})
	logEntry.Debug("Operation completed")

	statName := fmt.Sprintf("worker.%s.%s", operation, result)
	err := wo.Stats.TimingDuration(statName, duration, 1)
	if err != nil {
		logEntry.WithError(err).Warn("Error publishing detailed metrics to statsd")
	}
}

func (wo *Worker) monitorFileSize(fileSize int64) {
	logEntry := log.WithFields(log.Fields{
		"fileSize": fileSize,
	})
	logEntry.Debug("Upload file size")

	statName := "worker.upload-size"

	// Statsd gauge will not aggregate correctly, so we use Timing() and send the fileSize
	// instead of duration in milliseconds.
	err := wo.Stats.Timing(statName, fileSize, 1)
	if err != nil {
		logEntry.WithError(err).Warn("Error publishing file size metrics to statsd")
	}
}
