package worker

import (
	"encoding/json"
	"errors"
	"reflect"
	"testing"

	"code.justin.tv/web/upload-service/awsmocks"
	"code.justin.tv/web/upload-service/backend/mocks"
	"code.justin.tv/web/upload-service/models"

	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type MonitoringTestSuite struct {
	suite.Suite
	backender *mocks.Backender
	worker    *Worker
}

func (suite *MonitoringTestSuite) SetupTest() {
	suite.backender = &mocks.Backender{}
	suite.worker = &Worker{Backend: suite.backender}
}

func (suite *MonitoringTestSuite) TestMonitoringSNS() {
	message := &uploader.StatusResponse{
		UploadId: "upload ID",
		Status:   uploader.Status_PROCESSING_FAILED,
		Message:  "returned error",
	}
	messageJSON, err := json.Marshal(message)
	suite.Require().NoError(err)
	expected := sns.PublishInput{
		Message:  aws.String(string(messageJSON)),
		TopicArn: aws.String("sns topic"),
	}

	matchFn := func(in *sns.PublishInput) bool {
		return reflect.DeepEqual(in, &expected)
	}
	mockSNS := &awsmocks.SNSAPI{}
	mockSNS.On("Publish", mock.MatchedBy(matchFn)).Return(nil, nil)
	suite.backender.On("SNS").Return(mockSNS)

	suite.worker.processUploadMonitoring(models.Monitoring{SNSTopic: "sns topic"}, monitoringResult{
		retErr:   errors.New("returned error"),
		uploadID: "upload ID",
	})
	mockSNS.AssertNumberOfCalls(suite.T(), "Publish", 1)
}

func TestMonitoring(t *testing.T) {
	suite.Run(t, new(MonitoringTestSuite))
}
