package worker

import "github.com/heroku/rollbar"

type rollbarMsg struct {
	token  string
	err    error
	extras map[string]interface{}
}

// RollbarWrapper defines structs that implment the functionality we need,
// namely the ability to couple tokens with rollbar messages to prevent race conditions.
type RollbarWrapper interface {
	QueueRollbar(token string, err error, uploadID string)
}

// AsyncRollbar defines structs that implement the specific methods that we require
// from heroku/rollbar.AsyncClient
type AsyncRollbar interface {
	Wait()
	SetToken(token string)
	SetCustom(custom map[string]interface{})
	ErrorWithExtras(level string, err error, extras map[string]interface{})
}

// NewRollbarWrapper returns a RollbarWrapper from a concrete heroku/rollbar.AsyncClient
func NewRollbarWrapper(client *rollbar.AsyncClient) RollbarWrapper {
	if client == nil {
		return nil
	}
	return NewAbstractRollbarWrapper(client, client.Buffer)
}

// NewAbstractRollbarWrapper returns a RollbarWrapper from any struct or mock that
// implements the functions we use. Since AsyncClient.Buffer does not have a getter method,
// the buffer must be passed in explicitly
func NewAbstractRollbarWrapper(client AsyncRollbar, buffer int) RollbarWrapper {
	r := &rollbarWrapper{
		client: client,
		queue:  make(chan rollbarMsg, buffer),
	}
	go func() {
		for msg := range r.queue {
			r.client.Wait()
			r.client.SetToken(msg.token)
			r.client.SetCustom(make(map[string]interface{}))
			r.client.ErrorWithExtras(rollbar.ERR, msg.err, msg.extras)
		}
	}()
	return r
}

// rollbarWrapper is a concrete implementation of RollbarWrapper. It should not be instantiated directly,
// instead use NewAbstractRollbarWrapper
type rollbarWrapper struct {
	client AsyncRollbar
	queue  chan rollbarMsg
}

func (r *rollbarWrapper) QueueRollbar(token string, err error, uploadID string) {
	r.queue <- rollbarMsg{token, err, map[string]interface{}{
		"uploadID": uploadID,
	}}
}
