package worker

import (
	"fmt"
	"math"
	"net/http"

	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"

	"os"

	"code.justin.tv/web/upload-service/transformations"
)

type ValidationError struct {
	status uploader.Status

	expected, actual string
}

func (e *ValidationError) Error() string {
	str := e.status.String()
	if e.expected != "" && e.actual != "" {
		str += " expected: " + e.expected + ", actual: " + e.actual
	}
	return str
}

func (e *ValidationError) Status() uploader.Status {
	return e.status
}

type Validator interface {
	Validate(info *transformations.FileInfo) StatusError
}

type aspectRatioValidation models.Constraint

func (v aspectRatioValidation) Validate(info *transformations.FileInfo) StatusError {
	if info.Height == 0 {
		return nil
	}

	ratio := float64(info.Width) / float64(info.Height)
	if !satisfyConstraint(models.Constraint(v), ratio) {
		return v.makeError(ratio)
	}
	return nil
}

func (v aspectRatioValidation) makeError(actual float64) StatusError {
	return &ValidationError{
		uploader.Status_ASPECT_RATIO_VALIDATION_FAILED,
		fmt.Sprintf("%s%.5f", v.Test, v.Value),
		fmt.Sprintf("%.5f", actual),
	}
}

type widthValidation models.Constraint

func (v widthValidation) Validate(info *transformations.FileInfo) StatusError {
	if !satisfyConstraint(models.Constraint(v), float64(info.Width)) {
		return v.makeError(info.Width)
	}
	return nil
}

func (v widthValidation) makeError(actual uint) StatusError {
	return &ValidationError{uploader.Status_WIDTH_VALIDATION_FAILED, fmt.Sprintf("%s%.5f", v.Test, v.Value), fmt.Sprintf("%d", actual)}
}

type heightValidation models.Constraint

func (v heightValidation) Validate(info *transformations.FileInfo) StatusError {
	if !satisfyConstraint(models.Constraint(v), float64(info.Height)) {
		return v.makeError(info.Height)
	}
	return nil
}

func (v heightValidation) makeError(actual uint) StatusError {
	return &ValidationError{uploader.Status_HEIGHT_VALIDATION_FAILED, fmt.Sprintf("%s%.5f", v.Test, v.Value), fmt.Sprintf("%d", actual)}
}

type imageFormatValidation struct {
	Expected string
	File     *os.File
}

func (v imageFormatValidation) Validate(info *transformations.FileInfo) StatusError {
	if !info.IsImage {
		return v.makeError(uploader.Status_IS_IMAGE_VALIDATION_FAILED, guessFiletype(v.File))
	}

	if v.Expected == "image" {
		return nil
	}

	// normalize jpeg spelling into jpg.
	if v.Expected == "jpeg" {
		v.Expected = "jpg"
	}
	if info.Format != v.Expected {
		return v.makeError(uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED, info.Format)
	}
	return nil
}

func (v imageFormatValidation) makeError(status uploader.Status, actual string) StatusError {
	return &ValidationError{status, v.Expected, actual}
}

type fileSizeValidation uint64

func (v fileSizeValidation) Validate(info *transformations.FileInfo) StatusError {
	return v.ValidateSize(info.Size)
}

func (v fileSizeValidation) ValidateSize(actual int64) StatusError {
	if v == 0 || actual <= int64(v) {
		return nil
	}

	return v.makeError(actual)
}

func (v fileSizeValidation) makeError(actual int64) StatusError {
	return &ValidationError{uploader.Status_FILE_SIZE_VALIDATION_FAILED, fmt.Sprintf("<= %d bytes", v), fmt.Sprintf("%d bytes", actual)}
}

func satisfyConstraint(constraint models.Constraint, actual float64) bool {
	switch constraint.Test {
	case "=":
		return math.Abs(actual-constraint.Value) <= .000001
	case "<":
		return actual < constraint.Value
	case "<=":
		return actual <= constraint.Value
	case ">":
		return actual > constraint.Value
	case ">=":
		return actual >= constraint.Value
	default:
		return true
	}
}

func guessFiletype(tmpfile *os.File) string {
	if tmpfile == nil {
		return "non-image"
	}

	_, err := tmpfile.Seek(0, 0)
	if err != nil {
		return "non-image"
	}

	buf := make([]byte, 512)
	_, err = tmpfile.Read(buf)
	if err != nil {
		return "non-image"
	}

	detected := http.DetectContentType(buf)
	return detected
}
