package worker

import (
	"os"
	"testing"

	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"

	"github.com/stretchr/testify/suite"
)

type NoopTransformer struct {
	info transformations.FileInfo
}

func (n *NoopTransformer) Initialize() {}
func (n *NoopTransformer) Terminate()  {}
func (n *NoopTransformer) RunTransformations(filename, tmpDir, idx string, info *transformations.FileInfo, transformations []transformations.Transformation) (*os.File, error) {
	return nil, nil
}
func (n *NoopTransformer) FillInfoFromFile(tmpfile *os.File, info *transformations.FileInfo) error {
	if info.IsImage {
		return nil
	}
	info.IsImage = n.info.IsImage
	info.Size = n.info.Size
	info.SetFormat(n.info.Format)
	info.SetWidth(n.info.Width)
	info.SetHeight(n.info.Height)
	return nil
}

type ValidationsTestSuite struct {
	suite.Suite
	worker Worker
}

func (suite *ValidationsTestSuite) SetupTest() {
	suite.worker = Worker{Transformer: &NoopTransformer{}}

	suite.setupInfo(transformations.FileInfo{
		IsImage: true,
		Size:    int64(13016),
		Width:   uint(195),
		Height:  uint(209),
		Format:  "jpg",
	})
}

func (suite *ValidationsTestSuite) setupInfo(info transformations.FileInfo) {
	suite.worker.Transformer.(*NoopTransformer).info = info
}

func (suite *ValidationsTestSuite) TestReplaceParamsNoProcessing() {
	info := transformations.NewFileInfo("upload-id")

	err := suite.worker.validate(nil, info, models.Validation{})
	suite.Require().NoError(err)

	result := info.ExecTemplate("asdf{{UploadID}}_{{UploadID}}__{{Height}}x{{Width}}_{{Dimensions}}{{Extension}}")
	suite.Require().Equal("asdfupload-id_upload-id__x_", result)
}

func (suite *ValidationsTestSuite) TestValidationsPassing() {
	validation := models.Validation{
		Format:           "jpg",
		FileSizeLessThan: 14 << 10,
		AspectRatioConstraints: []models.Constraint{
			{195.0 / 209.0, "="},
		},
		WidthConstraints: []models.Constraint{
			{190, ">="},
			{200, "<="},
		},
		HeightConstraints: []models.Constraint{
			{205, ">="},
			{210, "<="},
		},
	}

	info := transformations.NewFileInfo("upload-id")
	err := suite.worker.validate(nil, info, validation)
	suite.Require().NoError(err)
}

func (suite *ValidationsTestSuite) TestFillInInfoParams() {
	validation := models.Validation{
		Format:           "image",
		FileSizeLessThan: 14 << 10,
	}

	info := transformations.NewFileInfo("upload-id")
	err := suite.worker.validate(nil, info, validation)
	suite.Require().NoError(err)

	suite.Require().Equal("upload-id", info.UploadID)
	suite.Require().Equal(true, info.IsImage)
	suite.Require().Equal(int64(13016), info.Size)
	suite.Require().Equal(uint(195), info.Width)
	suite.Require().Equal(uint(209), info.Height)
	suite.Require().Equal("jpg", info.Format)

	result := info.ExecTemplate("asdf{{UploadID}}_{{UploadID}}__{{Height}}x{{Width}}_{{Dimensions}}{{Extension}}")
	suite.Require().Equal("asdfupload-id_upload-id__209x195_195x209.jpg", result)
}

func (suite *ValidationsTestSuite) TestValidationFailures() {
	equalConstraint := models.Constraint{196.0 / 209.0, "="}
	gtConstraint := models.Constraint{209, ">"}
	geqConstraint := models.Constraint{205, ">="}
	ltConstraint := models.Constraint{195, "<"}
	leqConstraint := models.Constraint{200, "<="}

	validationErrors := []struct {
		validation models.Validation
		msg        string
		status     uploader.Status
	}{
		{models.Validation{FileSizeLessThan: 100},
			"FILE_SIZE_VALIDATION_FAILED expected: <= 100 bytes, actual: 13016 bytes",
			uploader.Status_FILE_SIZE_VALIDATION_FAILED,
		},
		{models.Validation{AspectRatioConstraints: []models.Constraint{equalConstraint}},
			"ASPECT_RATIO_VALIDATION_FAILED expected: =0.93780, actual: 0.93301",
			uploader.Status_ASPECT_RATIO_VALIDATION_FAILED,
		},
		{models.Validation{WidthConstraints: []models.Constraint{geqConstraint}},
			"WIDTH_VALIDATION_FAILED expected: >=205.00000, actual: 195",
			uploader.Status_WIDTH_VALIDATION_FAILED,
		},
		{models.Validation{WidthConstraints: []models.Constraint{ltConstraint}},
			"WIDTH_VALIDATION_FAILED expected: <195.00000, actual: 195",
			uploader.Status_WIDTH_VALIDATION_FAILED,
		},
		{models.Validation{HeightConstraints: []models.Constraint{gtConstraint}},
			"HEIGHT_VALIDATION_FAILED expected: >209.00000, actual: 209",
			uploader.Status_HEIGHT_VALIDATION_FAILED,
		},
		{models.Validation{HeightConstraints: []models.Constraint{leqConstraint}},
			"HEIGHT_VALIDATION_FAILED expected: <=200.00000, actual: 209",
			uploader.Status_HEIGHT_VALIDATION_FAILED,
		},
		{models.Validation{Format: "png"},
			"IMAGE_FORMAT_VALIDATION_FAILED expected: png, actual: jpg",
			uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED,
		},
	}

	for i, testCase := range validationErrors {
		info := transformations.NewFileInfo("upload-id")
		validationErr := suite.worker.validate(nil, info, testCase.validation)
		suite.Require().Equal(testCase.msg, validationErr.Error(), "test case %d", i)
		suite.Require().Equal(testCase.status, validationErr.Status(), "test case %d", i)
	}
}

func (suite *ValidationsTestSuite) TestSizeShortCircuit() {
	validation := models.Validation{
		Format:           "jpg",
		FileSizeLessThan: 100,
		AspectRatioConstraints: []models.Constraint{
			{195.0 / 209.0, "="},
		},
		WidthConstraints: []models.Constraint{
			{190, ">="},
			{200, "<="},
		},
		HeightConstraints: []models.Constraint{
			{205, ">="},
			{210, "<="},
		},
	}

	info := transformations.NewFileInfo("upload-id")
	validationErr := suite.worker.validate(nil, info, validation)
	suite.Require().Equal(fileSizeValidation(100).makeError(13016), validationErr)
	suite.Require().Equal(uploader.Status_FILE_SIZE_VALIDATION_FAILED, validationErr.Status())
}

func (suite *ValidationsTestSuite) TestIllegalImageValidation() {
	suite.setupInfo(transformations.FileInfo{IsImage: false})

	validations := []models.Validation{
		{AspectRatioConstraints: []models.Constraint{
			{195.0 / 209.0, "="},
		}},
		{WidthConstraints: []models.Constraint{
			{190, ">="},
			{200, "<="},
		}},
		{HeightConstraints: []models.Constraint{
			{205, ">="},
			{210, "<="},
		}},
		{Format: "png"},
	}

	for _, validation := range validations {
		info := transformations.NewFileInfo("upload-id")
		validationErr := suite.worker.validate(nil, info, validation)
		suite.Require().Equal(uploader.Status_IS_IMAGE_VALIDATION_FAILED, Status(validationErr))
	}
}

func (suite *ValidationsTestSuite) TestFormatValidation() {
	validation := models.Validation{
		Format: "good format",
	}
	info := transformations.NewFileInfo("upload-id")
	info.IsImage = true

	info.Format = "good format"
	err := suite.worker.validate(nil, info, validation)
	suite.Require().NoError(err)

	info.Format = "bad format"
	validationErr := suite.worker.validate(nil, info, validation)
	suite.Require().Equal(imageFormatValidation{"good format", nil}.makeError(uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED, "bad format"), validationErr)
	suite.Require().Equal(uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED, validationErr.Status())
}

func (suite *ValidationsTestSuite) TestWidthValidation() {
	validation := models.Validation{
		WidthConstraints: []models.Constraint{
			{100, ">="},
			{200, "<="},
		},
	}
	info := transformations.NewFileInfo("upload-id")
	info.IsImage = true

	cases := []struct {
		width uint
		err   StatusError
	}{
		{123, nil},
		{100, nil},
		{200, nil},
		{99, &ValidationError{
			uploader.Status_WIDTH_VALIDATION_FAILED,
			">=100.00000",
			"99",
		}},
		{201, &ValidationError{
			uploader.Status_WIDTH_VALIDATION_FAILED,
			"<=200.00000",
			"201",
		}},
	}

	for i, test := range cases {
		info.Width = test.width
		err := suite.worker.validate(nil, info, validation)
		suite.Require().Equal(test.err, err, "test case %d", i)
	}
}

func (suite *ValidationsTestSuite) TestHeightValidation() {
	validation := models.Validation{
		HeightConstraints: []models.Constraint{
			{100, ">="},
			{200, "<="},
		},
	}
	info := transformations.NewFileInfo("upload-id")
	info.IsImage = true

	cases := []struct {
		height uint
		err    StatusError
	}{
		{123, nil},
		{100, nil},
		{200, nil},
		{99, &ValidationError{
			uploader.Status_HEIGHT_VALIDATION_FAILED,
			">=100.00000",
			"99",
		}},
		{201, &ValidationError{
			uploader.Status_HEIGHT_VALIDATION_FAILED,
			"<=200.00000",
			"201",
		}},
	}

	for i, test := range cases {
		info.Height = test.height
		err := suite.worker.validate(nil, info, validation)
		suite.Require().Equal(test.err, err, "test case %d", i)
	}
}

func (suite *ValidationsTestSuite) TestAspectRatioValidation() {
	validation := models.Validation{
		AspectRatioConstraints: []models.Constraint{
			{909.0 / 900.0, "="},
		},
	}
	info := transformations.NewFileInfo("upload-id")
	info.IsImage = true

	cases := []struct {
		width, height uint
		err           StatusError
	}{
		{909, 900, nil},
		{707, 700, nil},
		{0, 0, nil},
		{708, 701, &ValidationError{
			uploader.Status_ASPECT_RATIO_VALIDATION_FAILED,
			"=1.01000",
			"1.00999",
		}},
		{706, 699, &ValidationError{
			uploader.Status_ASPECT_RATIO_VALIDATION_FAILED,
			"=1.01000",
			"1.01001",
		}},
	}

	for i, test := range cases {
		info.SetHeight(test.height)
		info.SetWidth(test.width)
		err := suite.worker.validate(nil, info, validation)
		suite.Require().Equal(test.err, err, "test case %d", i)
	}
}

func TestValidations(t *testing.T) {
	suite.Run(t, new(ValidationsTestSuite))
}
