package worker

import (
	"context"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"path"
	"strconv"
	"time"

	"code.justin.tv/web/upload-service/backend"
	"code.justin.tv/web/upload-service/files"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/eawsy/aws-lambda-go-event/service/lambda/runtime/event/s3evt"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type SNSNotification struct {
	Type, Message string
}

type Union struct {
	s3evt.Event
	SNSNotification
}

type Worker struct {
	Backend backend.Backender
	Stats   statsd.Statter

	Transformer transformations.ImageTransformer
	Files       files.FileOperations

	MonitoringStatter statsd.Statter
	MonitoringRollbar RollbarWrapper

	TwirpClient uploader.Uploader
}

func (wo *Worker) SQSRouter(w http.ResponseWriter, r *http.Request) {
	var body Union
	bodyText, err := ioutil.ReadAll(r.Body)
	if err == nil {
		err = json.Unmarshal(bodyText, &body)
	}

	var handler string

	// HTTP status code to be returned on error
	errorCode := http.StatusInternalServerError

	if err != nil {
	} else if body.Type == "Notification" {
		err = wo.UpdateStatus(r.Context(), w, body.SNSNotification)
		handler = "UpdateStatus"
	} else if len(body.Records) > 0 {
		err = wo.ProcessUpload(r.Context(), w, body.Event, r.Header.Get("x-aws-sqsd-receive-count"))
		// If the error is not retryable, return OK so that the message gets removed from sqs.
		if Status(err) != uploader.Status_POSTPROCESS_RETRYING {
			errorCode = http.StatusOK
		}
		handler = "ProcessUpload"
	} else {
		err = errors.New("Unknown message type")
		handler = "unknown"
	}

	if err != nil {
		entry := log.WithFields(log.Fields{
			"messageID":    r.Header.Get("x-aws-sqsd-msgid"),
			"receiveCount": r.Header.Get("x-aws-sqsd-receive-count"),
			"bodyStruct":   body,
			"fullBody":     string(bodyText),
			"handler":      handler,
		})

		if re, ok := err.(*RetryableError); ok {
			entry = entry.WithFields(log.Fields{
				"attempt":       re.AttemptNumber,
				"totalAttempts": re.TriesTotal,
				"uploadID":      re.UploadID,
			})
		}

		entry.WithError(err).Warn("Error processing SQS message")
		w.WriteHeader(errorCode)

		w.Write([]byte(err.Error()))
	} else {
		w.Write([]byte("OK"))
	}
}

func (wo *Worker) UpdateStatus(ctx context.Context, w http.ResponseWriter, event SNSNotification) error {
	var eventData models.SNSCallback
	err := json.Unmarshal([]byte(event.Message), &eventData)
	if eventData.UploadID == "" {
		errMsg := "Unable to parse upload_id from " + event.Message
		if err != nil {
			return errors.Wrap(err, errMsg)
		}
		return errors.New(errMsg)
	}

	status := uploader.Status(eventData.Status)
	if status != uploader.Status_POSTPROCESS_COMPLETE {
		log.WithFields(log.Fields{
			"status":   status.String(),
			"uploadID": eventData.UploadID,
		}).Info("Not changing status for SNS Callback")
		return nil
	}

	request := &uploader.SetStatusRequest{
		UploadId: eventData.UploadID,
		Status:   uploader.Status_COMPLETE,
	}

	_, err = wo.TwirpClient.SetStatus(ctx, request)
	if err != nil {
		return err
	}

	return nil
}

func (wo *Worker) ProcessUpload(ctx context.Context, w http.ResponseWriter, event s3evt.Event, receiveCount string) error {
	count, countErr := strconv.Atoi(receiveCount)

	if countErr != nil {
		log.WithFields(log.Fields{
			"receiveCount": receiveCount,
		}).WithError(countErr).Info("Unable to parse receiveCount as an integer")

		count = -1
	}

	// 'event.Records' is a slice of records, but we are expecting that it contains at most one.
	if len(event.Records) > 1 {
		log.WithFields(log.Fields{
			"recordCount": len(event.Records),
		}).Warn("Worker received event with more than one record, dropping requests")
	} else if len(event.Records) == 0 {
		log.Info("Worker received empty event")
		return nil
	}

	record := event.Records[0]
	uploadID := path.Base(record.S3.Object.Key)
	nextStatus := uploader.Status_PROCESSING_FAILED

	log.WithFields(log.Fields{
		"uploadID":         uploadID,
		"sqsAttemptNumber": count,
	}).Info("Starting to process upload")

	upload, err := wo.Backend.GetMetadata(uploadID)
	if err != nil {
		if _, ok := err.(backend.UserError); ok {
			err = errors.Wrapf(err, "Backend GetMetadata could not retrieve upload %q", uploadID)
		}
		err = errors.Wrapf(err, "Backend GetMetadata failure for %q", uploadID)
	} else {
		setErr := wo.Backend.SetStatus(ctx, uploadID, uploader.Status_POSTPROCESS_STARTED, "")
		if setErr != nil {
			log.WithFields(log.Fields{
				"uploadID": uploadID,
			}).WithError(setErr).Warn("Error updating metadata to POSTPROCESS_STARTED")
		}

		startTime := time.Now()
		err = wo.processUpload(ctx, upload, record.S3.Bucket.Name, record.S3.Object.Key)
		nextStatus = Status(err)

		wo.processUploadMonitoring(upload.Monitoring, monitoringResult{uploadID, startTime, err})
	}

	statusMessage := ""
	if err != nil {
		statusMessage = err.Error()
	}

	if nextStatus == uploader.Status_TRANSFORMATION_FAILED ||
		nextStatus == uploader.Status_PROCESSING_FAILED {
		err = &RetryableError{
			AttemptNumber: count,
			TriesTotal:    MaxRetries,
			UploadID:      uploadID,
			Wraps:         err,
		}
		nextStatus = Status(err)
	}

	setErr := wo.Backend.SetStatus(ctx, uploadID, nextStatus, statusMessage)

	if setErr != nil {
		log.WithFields(log.Fields{
			"uploadID": uploadID,
			"status":   nextStatus,
			"attempt":  count,
		}).WithError(setErr).Warn("Error updating metadata status after processing")
	}

	// In case the upload has moved to a terminal failed state, make a callback to the feature service SNS topic and PubSub topic.
	if err != nil && nextStatus != uploader.Status_POSTPROCESS_RETRYING {
		startTime := time.Now()
		callbackErr := wo.Backend.NotifyCallbacks(ctx, upload, nextStatus, nil)

		logFields := log.Fields{
			"uploadID": uploadID,
			"status":   nextStatus,
		}
		if upload != nil {
			logFields["callback"] = upload.Callback
		}
		wo.monitorTiming("NotifyCallbacks", startTime, callbackErr != nil, logFields)
		if callbackErr != nil {
			log.WithFields(logFields).WithError(callbackErr).Error("Failed to make Callbacks for failed upload")
		}
	}

	return err
}
