package worker_test

import (
	"bytes"
	"encoding/json"
	"errors"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"testing"

	"code.justin.tv/web/upload-service/api"
	"code.justin.tv/web/upload-service/api/worker"
	"code.justin.tv/web/upload-service/backend/mocks"
	fileMocks "code.justin.tv/web/upload-service/files/mocks"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	transformMocks "code.justin.tv/web/upload-service/transformations/mocks"

	goji "goji.io"

	"fmt"

	"code.justin.tv/web/upload-service/transformations"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/eawsy/aws-lambda-go-event/service/lambda/runtime/event/s3evt"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type ProcessUploadTestSuite struct {
	suite.Suite
	backender   *mocks.Backender
	testServer  *httptest.Server
	transformer *transformMocks.ImageTransformer

	input  s3evt.Event
	upload *models.Upload
}

func (suite *ProcessUploadTestSuite) SetupTest() {
	suite.backender = &mocks.Backender{}
	suite.backender.On("EC2InstanceID").Return("")

	files := &fileMocks.FileOperations{}
	files.On("Remove", mock.Anything).Return(nil)
	files.On("TempDir", "", mock.Anything).Return("/path/to/tempdir", nil)
	files.On("RemoveAll", "/path/to/tempdir").Return(nil)

	suite.transformer = &transformMocks.ImageTransformer{}

	mux := goji.NewMux()
	api.NewServerWithWorker(mux, &worker.Worker{suite.backender, &statsd.NoopClient{}, suite.transformer, files, nil, nil, nil})
	suite.testServer = httptest.NewServer(mux)

	suite.input = s3evt.Event{
		Records: []*s3evt.EventRecord{
			{S3: &s3evt.Record{Object: &s3evt.Object{Key: "i-am-a-guid"}, Bucket: &s3evt.Bucket{Name: "i-am-a-bucket"}}},
		},
	}
	suite.upload = &models.Upload{
		Callback: models.UploadCallback{ARN: "i-am-an-arn", Data: []byte("callback data")},
		Outputs: []models.Output{
			{Name: "image.jpg"},
		},
		OutputPrefix: "s3://bucket/path/",
	}
}

func (suite *ProcessUploadTestSuite) TearDownTest() {
	suite.testServer.Close()
}

func (suite *ProcessUploadTestSuite) makeRequest(body io.Reader) *http.Response {
	return suite.makeRequestAttempt(body, "")
}

func (suite *ProcessUploadTestSuite) makeRequestAttempt(body io.Reader, attemptNumber string) *http.Response {
	req, err := http.NewRequest("POST", suite.testServer.URL+"/worker", body)
	suite.Require().NoError(err)

	req.Header.Set("x-aws-sqsd-receive-count", attemptNumber)

	resp, err := http.DefaultClient.Do(req)
	suite.Require().NoError(err)

	return resp
}

// Given good input, we should read metadata and publish to SNS
func (suite *ProcessUploadTestSuite) TestProcessUpload() {
	suite.backender.On("GetMetadata", "i-am-a-guid").Return(suite.upload, nil).Once()

	tmpfile, err := ioutil.TempFile("", "tmpS3File")
	suite.Require().NoError(err)
	defer os.Remove(tmpfile.Name())

	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_STARTED, "").Return(nil).Once()
	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_COMPLETE, "").Return(nil).Once()
	suite.backender.On("DownloadS3", mock.Anything, mock.Anything).Return(tmpfile, nil).Once()
	suite.backender.On("UploadS3", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil).Once()
	suite.backender.On("FileSizeS3", mock.Anything, mock.Anything).Return(int64(1), nil).Once()
	suite.backender.On("NotifyCallbacks", mock.Anything, suite.upload, uploader.Status_POSTPROCESS_COMPLETE, mock.Anything).Return(nil)

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	resp := suite.makeRequest(bytes.NewReader(data))
	suite.Require().NoError(err)
	suite.Require().Equal(200, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

// When unable to get metadata, we should retry
func (suite *ProcessUploadTestSuite) TestProcessUploadNoMetadataRetrying() {
	suite.backender.On("GetMetadata", "i-am-a-guid").Return(nil, errors.New("I am an error")).Once()
	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_RETRYING, mock.Anything).Return(nil).Once()

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	resp := suite.makeRequestAttempt(bytes.NewReader(data), "1")
	suite.Require().NoError(err)
	suite.Require().NotEqual(200, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

// When unable to get metadata and are out of retries we should fail the request and not publish to SNS
func (suite *ProcessUploadTestSuite) TestProcessUploadNoMetadataNotRetrying() {
	suite.backender.On("GetMetadata", "i-am-a-guid").Return(nil, errors.New("I am an error")).Once()

	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_PROCESSING_FAILED, mock.Anything).Return(nil).Once()
	suite.backender.On("NotifyCallbacks", mock.Anything, (*models.Upload)(nil), uploader.Status_PROCESSING_FAILED, []uploader.OutputInfo(nil)).Return(nil)

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	resp := suite.makeRequestAttempt(bytes.NewReader(data), fmt.Sprintf("%d", worker.MaxRetries))
	suite.Require().NoError(err)
	suite.Require().Equal(200, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

// When uploader doesn't specify an SNS topic ARN, we shouldn't die
func (suite *ProcessUploadTestSuite) TestProcessUploadNoCallback() {
	suite.upload.Callback = models.UploadCallback{ARN: "NULL"}
	suite.backender.On("GetMetadata", "i-am-a-guid").Return(suite.upload, nil).Once()

	tmpfile, err := ioutil.TempFile("", "tmpS3File")
	suite.Require().NoError(err)
	defer os.Remove(tmpfile.Name())

	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_STARTED, "").Return(nil).Once()
	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_COMPLETE, "").Return(nil).Once()
	suite.backender.On("FileSizeS3", mock.Anything, mock.Anything).Return(int64(1), nil).Once()
	suite.backender.On("DownloadS3", mock.Anything, mock.Anything).Return(tmpfile, nil).Once()
	suite.backender.On("UploadS3", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil).Once()
	suite.backender.On("NotifyCallbacks", mock.Anything, suite.upload, uploader.Status_POSTPROCESS_COMPLETE, mock.Anything).Return(nil)

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	resp := suite.makeRequest(bytes.NewReader(data))
	suite.Require().NoError(err)
	suite.Require().Equal(200, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

func (suite *ProcessUploadTestSuite) TestProcessFailedRetrying() {
	suite.backender.On("GetMetadata", "i-am-a-guid").Return(suite.upload, nil).Once()

	tmpfile, err := ioutil.TempFile("", "tmpS3File")
	suite.Require().NoError(err)
	defer os.Remove(tmpfile.Name())

	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_STARTED, "").Return(nil).Once()
	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_RETRYING, mock.Anything).Return(nil).Once()
	suite.backender.On("FileSizeS3", mock.Anything, mock.Anything).Return(int64(1), nil).Once()
	suite.backender.On("DownloadS3", mock.Anything, mock.Anything).Return(tmpfile, nil).Once()
	suite.backender.On("UploadS3", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(errors.New("err")).Once()
	// Unlike the other cases, no SNSCallback mock here because it shouldn't be called.

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	// Should be retried because we are not yet at the maximum number of retries.
	resp := suite.makeRequestAttempt(bytes.NewReader(data), "1")
	suite.Require().NoError(err)
	suite.Require().Equal(500, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

func (suite *ProcessUploadTestSuite) TestProcessFailedNotRetrying() {

	suite.backender.On("GetMetadata", "i-am-a-guid").Return(suite.upload, nil).Once()

	tmpfile, err := ioutil.TempFile("", "tmpS3File")
	suite.Require().NoError(err)
	defer os.Remove(tmpfile.Name())

	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_STARTED, "").Return(nil).Once()
	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_PROCESSING_FAILED, mock.Anything).Return(nil).Once()
	suite.backender.On("FileSizeS3", mock.Anything, mock.Anything).Return(int64(1), nil).Once()
	suite.backender.On("DownloadS3", mock.Anything, mock.Anything).Return(tmpfile, nil).Once()
	suite.backender.On("UploadS3", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(errors.New("err")).Once()
	suite.backender.On("NotifyCallbacks", mock.Anything, suite.upload, uploader.Status_PROCESSING_FAILED, []uploader.OutputInfo(nil)).Return(nil)

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	// Will not retry (200 status) because we are at the max number of attempts.
	resp := suite.makeRequestAttempt(bytes.NewReader(data), fmt.Sprintf("%d", worker.MaxRetries))
	suite.Require().NoError(err)
	suite.Require().Equal(200, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

// TestProcessFailedValidation tests the fact that validation errors are surfaced in the correct manner
func (suite *ProcessUploadTestSuite) TestProcessFailedValidation() {
	suite.upload.PreValidation = models.Validation{
		Format: "PSD",
	}
	suite.backender.On("GetMetadata", "i-am-a-guid").Return(suite.upload, nil).Once()

	tmpfile, err := ioutil.TempFile("", "tmpS3File")
	suite.Require().NoError(err)
	defer os.Remove(tmpfile.Name())

	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_POSTPROCESS_STARTED, "").Return(nil).Once()
	suite.backender.On("SetStatus", mock.Anything, "i-am-a-guid", uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED, mock.Anything).Return(nil).Once()
	suite.backender.On("FileSizeS3", mock.Anything, mock.Anything).Return(int64(1), nil).Once()
	suite.backender.On("DownloadS3", mock.Anything, mock.Anything).Return(tmpfile, nil).Once()
	suite.backender.On("NotifyCallbacks", mock.Anything, suite.upload, uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED, []uploader.OutputInfo(nil)).Return(nil)

	suite.transformer.On("FillInfoFromFile", tmpfile, mock.Anything).Run(func(arguments mock.Arguments) {
		info := arguments.Get(1).(*transformations.FileInfo)
		info.IsImage = true
		info.SetFormat("jpg")
	}).Return(nil)

	data, err := json.Marshal(suite.input)
	suite.Require().NoError(err)

	resp := suite.makeRequestAttempt(bytes.NewReader(data), "1")
	suite.Require().NoError(err)
	suite.Require().Equal(200, resp.StatusCode)

	suite.backender.AssertExpectations(suite.T())
}

func TestProcessUpload(t *testing.T) {
	suite.Run(t, new(ProcessUploadTestSuite))
}
