package worker_test

import (
	"bytes"
	"encoding/json"
	"io"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"code.justin.tv/web/upload-service/api"
	"code.justin.tv/web/upload-service/api/worker"
	"code.justin.tv/web/upload-service/backend/mocks"
	"code.justin.tv/web/upload-service/models"
	twirpMocks "code.justin.tv/web/upload-service/rpc/server/mocks"

	goji "goji.io"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/suite"
)

type WorkerTestSuite struct {
	suite.Suite
	backender  *mocks.Backender
	testServer *httptest.Server

	twirpMock *twirpMocks.Uploader
}

func (suite *WorkerTestSuite) SetupTest() {
	suite.backender = &mocks.Backender{}
	suite.backender.On("EC2InstanceID").Return("")

	suite.twirpMock = &twirpMocks.Uploader{}
	wo := &worker.Worker{suite.backender, &statsd.NoopClient{}, nil, nil, nil, nil, suite.twirpMock}

	mux := goji.NewMux()
	api.NewServerWithWorker(mux, wo)

	suite.testServer = httptest.NewServer(mux)
}

func (suite *WorkerTestSuite) TearDownTest() {
	suite.testServer.Close()
}

func (suite *WorkerTestSuite) makeRequest(body io.Reader) *http.Response {
	resp, err := http.Post(
		suite.testServer.URL+"/worker",
		"application/json",
		body,
	)

	suite.Require().NoError(err)
	return resp
}

// The request body is invalid json, worker should return 400.
func (suite *WorkerTestSuite) TestSQSRouterInvalidBody() {
	resp := suite.makeRequest(strings.NewReader("xx"))
	suite.Require().NotEqual(200, resp.StatusCode)
}

// If the request is empty, we can't process it.
func (suite *WorkerTestSuite) TestSQSRouterNoEvents() {
	resp := suite.makeRequest(strings.NewReader("{}"))
	suite.Require().NotEqual(200, resp.StatusCode)
}

// If Message is not valid JSON, return 400
func (suite *WorkerTestSuite) TestUpdateStatusMalformed() {
	resp := suite.makeRequest(strings.NewReader(`{"Type": "Notification", "Message": "xx"}`))
	suite.Require().NotEqual(200, resp.StatusCode)
}

// If Message doesn't contain an UploadID, return 400
func (suite *WorkerTestSuite) TestUpdateStatusNoUploadID() {
	resp := suite.makeRequest(strings.NewReader(`{"Type": "Notification", "Message": "{}"}`))
	suite.Require().NotEqual(200, resp.StatusCode)
}

func (suite *WorkerTestSuite) TestUpdateStatusValidInput() {
	suite.twirpMock.MockSuccessSetStatus()

	message, err := json.Marshal(models.SNSCallback{UploadID: "upload-id"})
	suite.Require().NoError(err)
	request := map[string]string{
		"Type":    "Notification",
		"Message": string(message),
	}
	requestJSON, err := json.Marshal(request)
	suite.Require().NoError(err)

	resp := suite.makeRequest(bytes.NewReader(requestJSON))
	suite.Require().Equal(200, resp.StatusCode)

}

func TestWorker(t *testing.T) {
	suite.Run(t, new(WorkerTestSuite))
}
