package backend

import (
	"bufio"
	"context"
	"io"
	"io/ioutil"
	"net/url"
	"os"
	"time"

	log "github.com/sirupsen/logrus"

	pubclient "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
	"github.com/pkg/errors"
)

const (
	URLValidDuration   = 15 * time.Minute
	DefaultRegion      = "us-west-2"
	BucketOwnerControl = "bucket-owner-full-control"
)

func init() {
	config.Register(map[string]string{
		"METADATA_TABLE":  "",
		"S3_BUCKET":       "",
		"PUBSUB_ENDPOINT": "",
		"TOPIC_PREFIX":    "",
	})
}

type Backender interface {
	CreateMetadata(models.Upload) error
	CreatePresignedUrl(uploadId string) (string, error)
	GetMetadata(uploadID string) (*models.Upload, error)
	SetStatus(ctx context.Context, uploadID string, status uploader.Status, statusMessage string) error
	MetadataTable() string
	NotifyCallbacks(ctx context.Context, upload *models.Upload, status uploader.Status, outputInfos []uploader.OutputInfo) error

	S3() s3iface.S3API
	SNS() snsiface.SNSAPI
	DynamoDB() dynamodbiface.DynamoDBAPI
	EC2InstanceID() string
	PubClient() pubclient.PubClient

	FileSizeS3(bucket, key string) (int64, error)
	DownloadS3(bucket, key string) (*os.File, error)
	UploadS3(tmpfile io.ReadSeeker, outputPath, contentType, grantFullControl, grantRead string) error
	SNSCallback(arn string, data models.SNSCallback) error
	PubsubCallback(ctx context.Context, topic string, data models.SNSCallback, retries int) error
}

type Backend struct {
	s3Ingest      s3iface.S3API
	s3            s3iface.S3API
	sns           snsiface.SNSAPI
	dynamodb      dynamodbiface.DynamoDBAPI
	ec2metadata   *ec2metadata.EC2Metadata
	pubclient     pubclient.PubClient
	metadataTable string
	ingestBucket  string
	topicPrefix   string
}

func NewBackendFromConfigTableBucket(conf *aws.Config, table, bucket string, accelerate bool) (*Backend, error) {
	endpoint := config.Resolve("PUBSUB_ENDPOINT")
	clientConfig := twitchhttp.ClientConf{
		Host:  endpoint,
		Stats: config.Statsd(),
	}
	sess := session.Must(session.NewSession(conf))

	pubClient, err := pubclient.NewPubClient(clientConfig)

	if err != nil {
		log.WithError(err).WithField("endpoint", endpoint).Error("Failed to initialize pubsub")
		return nil, err
	}

	return &Backend{
		s3Ingest:      s3.New(sess, aws.NewConfig().WithS3UseAccelerate(accelerate)),
		s3:            s3.New(sess),
		sns:           sns.New(sess),
		dynamodb:      dynamodb.New(sess),
		ec2metadata:   ec2metadata.New(sess),
		metadataTable: table,
		ingestBucket:  bucket,
		pubclient:     pubClient,
		topicPrefix:   config.Resolve("TOPIC_PREFIX"),
	}, nil
}

func NewBackendFromConfig(conf *aws.Config) (*Backend, error) {
	return NewBackendFromConfigTableBucket(conf, config.Resolve("METADATA_TABLE"), config.Resolve("S3_BUCKET"), true)
}

func NewBackend() (*Backend, error) {
	return NewBackendFromConfig(&aws.Config{Region: aws.String(DefaultRegion)})
}

func (b *Backend) CreatePresignedUrl(uploadId string) (string, error) {
	req, _ := b.s3Ingest.PutObjectRequest(&s3.PutObjectInput{
		Bucket: aws.String(b.ingestBucket),
		Key:    aws.String(uploadId),
	})

	return req.Presign(URLValidDuration)
}

func (b *Backend) S3() s3iface.S3API {
	return b.s3
}

func (b *Backend) SNS() snsiface.SNSAPI {
	return b.sns
}

func (b *Backend) DynamoDB() dynamodbiface.DynamoDBAPI {
	return b.dynamodb
}
func (b *Backend) EC2InstanceID() string {
	if b.ec2metadata.Available() {
		doc, err := b.ec2metadata.GetInstanceIdentityDocument()
		if err != nil {
			log.WithError(err).Warn("Could not retrieve EC2 instance ID")
			return "unknown"
		}
		return doc.InstanceID
	} else {
		return "unavailable"
	}
}

func (b *Backend) MetadataTable() string {
	return b.metadataTable
}

func (b *Backend) PubClient() pubclient.PubClient {
	return b.pubclient
}

func (b *Backend) FileSizeS3(bucket, key string) (int64, error) {
	headObject, err := b.S3().HeadObject(&s3.HeadObjectInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(key),
	})
	if err != nil {
		return 0, errors.Wrapf(err, "Error getting file size for %s/%s", bucket, key)
	}

	return *headObject.ContentLength, nil
}

func (b *Backend) DownloadS3(bucket, key string) (*os.File, error) {
	gotObject, err := b.S3().GetObject(&s3.GetObjectInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(key),
	})
	if err != nil {
		return nil, errors.Wrapf(err, "Error downloading file from %s/%s", bucket, key)
	}

	tmpfile, err := ioutil.TempFile("", "tmpS3File")
	if err != nil {
		return nil, errors.Wrapf(err, "Error saving temp file from %s/%s", bucket, key)
	}

	if _, err := bufio.NewReader(gotObject.Body).WriteTo(tmpfile); err != nil {
		os.Remove(tmpfile.Name())
		return nil, errors.Wrapf(err, "Error writing temp file from %s/%s", bucket, key)
	}
	return tmpfile, nil
}

func (b *Backend) UploadS3(tmpfile io.ReadSeeker, outputPath, contentType, grantFullControl, grantRead string) error {
	outputURL, err := url.Parse(outputPath)
	if err != nil {
		return errors.Wrapf(err, "Could not parse output path %s", outputPath)
	}
	outputBucket := outputURL.Hostname()
	outputKey := outputURL.Path

	putObjectInput := &s3.PutObjectInput{
		Body:   tmpfile,
		Bucket: aws.String(outputBucket),
		Key:    aws.String(outputKey),
	}
	if contentType != "" {
		putObjectInput.SetContentType(contentType)
	}
	if grantFullControl != "" {
		putObjectInput.SetGrantFullControl(grantFullControl)
	}
	if grantRead != "" {
		putObjectInput.SetGrantRead(grantRead)
	}
	if grantFullControl == "" && grantRead == "" {
		putObjectInput.SetACL(BucketOwnerControl)
	}
	_, err = b.S3().PutObject(putObjectInput)
	if err != nil {
		return errors.Wrapf(err, "Error uploading output to %s", outputPath)
	}
	return nil
}
