package backend

import (
	"io/ioutil"
	"os"
	"strings"
	"testing"

	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	"code.justin.tv/common/config"
	"code.justin.tv/web/upload-service/awsmocks"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
)

type BackendTestSuite struct {
	suite.Suite

	s3      *awsmocks.S3API
	backend *Backend
}

func (suite *BackendTestSuite) SetupTest() {
	suite.s3 = &awsmocks.S3API{}
	suite.backend = &Backend{
		s3: suite.s3,
	}
}

func (suite *BackendTestSuite) TestDefaultBackend() {
	defaultTable := config.Resolve("METADATA_TABLE")
	defaultBucket := config.Resolve("S3_BUCKET")
	defaultBackend, err := NewBackend()
	suite.Require().NoError(err)

	suite.Require().Equal(defaultTable, defaultBackend.metadataTable)
	suite.Require().Equal(defaultBucket, defaultBackend.ingestBucket)
}

func (suite *BackendTestSuite) TestUploadS3Options() {
	inputOutputs := []struct {
		contentType string
		full        string
		read        string
		nils        [4]bool
	}{
		{"image/jpeg", "a", "b", [4]bool{false, false, false, true}},
		{"image/jpeg", "", "", [4]bool{false, true, true, false}},
		{"", "a", "", [4]bool{true, false, true, true}},
		{"", "", "b", [4]bool{true, true, false, true}},
		{"", "", "", [4]bool{true, true, true, false}},
	}
	_ = inputOutputs

	for _, inOut := range inputOutputs {
		matchFn := func(in *s3.PutObjectInput) bool {
			nils := [4]bool{in.ContentType == nil, in.GrantFullControl == nil, in.GrantRead == nil, in.ACL == nil}
			return inOut.nils == nils
		}

		suite.s3 = &awsmocks.S3API{}
		suite.s3.On("PutObject", mock.MatchedBy(matchFn)).Return(nil, nil)
		suite.backend = &Backend{
			s3: suite.s3,
		}

		err := suite.backend.UploadS3(nil, "s3://bucket/path", inOut.contentType, inOut.full, inOut.read)
		suite.Require().NoError(err)

		suite.s3.AssertExpectations(suite.T())
	}
}

func (suite *BackendTestSuite) TestFileSizeS3() {
	suite.s3.On("HeadObject", mock.Anything).Return(&s3.HeadObjectOutput{ContentLength: aws.Int64(13)}, nil)
	size, err := suite.backend.FileSizeS3("bucket", "key")
	suite.Require().NoError(err)
	suite.Require().Equal(int64(13), size)
}

func (suite *BackendTestSuite) TestDownloadS3() {
	dataReadCloser := ioutil.NopCloser(strings.NewReader("arbitrary data"))
	gotObject := &s3.GetObjectOutput{
		Body: dataReadCloser,
	}

	suite.s3.On("GetObject", mock.Anything).Return(gotObject, nil)
	tmpFile, err := suite.backend.DownloadS3("bucket", "key")
	defer os.Remove(tmpFile.Name())

	suite.Require().NoError(err)

	all, err := ioutil.ReadFile(tmpFile.Name())
	suite.Require().NoError(err)

	suite.Require().Equal("arbitrary data", string(all))
}

func TestBackend(t *testing.T) {
	suite.Run(t, new(BackendTestSuite))
}
