package backend

import (
	"context"
	"encoding/json"

	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	Disabled = "disabled"
	Default  = "default"
	Empty    = ""

	pubsubRetries = 3
)

func (b *Backend) NotifyCallbacks(ctx context.Context, upload *models.Upload, status uploader.Status, outputInfos []uploader.OutputInfo) error {
	if upload == nil {
		return nil
	}

	callbackData := models.SNSCallback{
		UploadID: upload.UploadId,
		Outputs:  outputInfos,
		Data:     upload.Callback.Data,
		Status:   int64(status),
	}

	if err := b.SNSCallback(upload.Callback.ARN, callbackData); err != nil {
		return err
	}
	return b.PubsubCallback(ctx, upload.Callback.PubsubTopic, callbackData, pubsubRetries)
}

func (b *Backend) SNSCallback(arn string, data models.SNSCallback) error {
	if arn != "" && arn != "NULL" {
		messageJSON, err := json.Marshal(data)
		if err != nil {
			return errors.Wrapf(err, "Error while marshaling SNSCallback: %v", data)
		}

		publishInput := sns.PublishInput{
			Message:  aws.String(string(messageJSON)),
			TopicArn: aws.String(arn),
		}
		_, err = b.SNS().Publish(&publishInput)

		if err != nil {
			return errors.Wrapf(err, "Error while publishing status to %s", arn)
		}
	} else {
		log.WithFields(log.Fields{
			"uploadID": data.UploadID,
		}).Warn("No callback ARN specified")
	}
	return nil
}

func pubsubDisabled(topic string) bool {
	return topic == Disabled || topic == Empty
}

func (b *Backend) topicOrDefault(topic string, data models.SNSCallback) string {
	if topic == Default {
		return b.topicPrefix + data.UploadID
	}
	return topic
}

func (b *Backend) PubsubCallback(ctx context.Context, topic string, data models.SNSCallback, retries int) error {
	if pubsubDisabled(topic) {
		return nil
	}

	topics := []string{b.topicOrDefault(topic, data)}
	message, err := json.Marshal(data)

	if err != nil {
		return errors.Wrapf(err, "Error while marshaling SNSCallback: %v", data)
	}

	for k := 0; k < retries; k++ {
		if err = b.PubClient().Publish(ctx, topics, string(message), nil); err == nil {
			break
		}
	}

	log.WithFields(log.Fields{
		"uploadID": data.UploadID,
		"err":      err,
		"topics":   topics,
	}).Debug("published to pubsub")
	return err
}
