package backend

import (
	"errors"
	"testing"

	"context"

	"encoding/json"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/upload-service/awsmocks"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/pubclient/mocks"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type MessagesTestSuite struct {
	suite.Suite
}

func (suite *MessagesTestSuite) realBackend() Backender {
	res, err := NewBackend()
	suite.Require().NoError(err)
	return res
}

// NotifyCallbacks gently handles nil upload pointer
func (suite *MessagesTestSuite) TestNilPointer() {
	backend := suite.realBackend()

	err := backend.NotifyCallbacks(context.Background(), nil, uploader.Status_COMPLETE, nil)
	suite.Require().NoError(err)
}

func (suite *MessagesTestSuite) TestSNS() {
	topics := []string{Disabled, Empty}
	t := suite.T()

	for _, topic := range topics {
		t.Run("topic="+topic, func(innerT *testing.T) {
			suite.SetT(innerT)
			suite.testSNS(topic)
		})
	}
}

func (suite *MessagesTestSuite) testSNS(pubsubTopic string) {
	backend := &Backend{}
	backend.sns = &awsmocks.SNSAPI{}
	backend.pubclient = &mocks.PubClient{}

	ctx := context.WithValue(context.Background(), "test", true)
	upload := &models.Upload{
		UploadId: "upload-id",
		Callback: models.UploadCallback{
			ARN:         "sns-arn",
			Data:        []byte("data"),
			PubsubTopic: pubsubTopic,
		},
	}
	outputs := []uploader.OutputInfo{{Path: "foo"}}

	backend.sns.(*awsmocks.SNSAPI).On("Publish", mock.Anything).Once().Run(func(args mock.Arguments) {
		input := args.Get(0).(*sns.PublishInput)
		suite.Require().Equal(*input.TopicArn, upload.Callback.ARN)

		data := &models.SNSCallback{}
		err := json.Unmarshal([]byte(*input.Message), data)
		suite.Require().NoError(err)

		expectedData := models.SNSCallback{
			UploadID: upload.UploadId,
			Data:     upload.Callback.Data,
			Outputs:  outputs,
			Status:   int64(uploader.Status_COMPLETE),
		}
		suite.Require().Equal(*data, expectedData)
	}).Return(&sns.PublishOutput{}, nil)

	err := backend.NotifyCallbacks(ctx, upload, uploader.Status_COMPLETE, outputs)
	suite.Require().NoError(err)
}

func (suite *MessagesTestSuite) TestPubsub() {
	cases := []struct {
		givenTopic    string
		expectedTopic string
		numFailures   int
	}{
		{Default, "prefix.upload-id", 0},
		{Default, "prefix.upload-id", 2},
		{"foo.custom-topic", "foo.custom-topic", 0},
		{"foo.custom-topic", "foo.custom-topic", 2},
	}

	t := suite.T()

	for _, testCase := range cases {
		t.Run("topic="+testCase.givenTopic, func(innerT *testing.T) {
			suite.SetT(innerT)
			suite.testPubsub(testCase.givenTopic, testCase.expectedTopic, testCase.numFailures)
		})
	}
}

func (suite *MessagesTestSuite) testPubsub(givenTopic, expectedTopic string, numFailures int) {
	backend := &Backend{topicPrefix: "prefix."}
	backend.sns = &awsmocks.SNSAPI{}
	backend.pubclient = &mocks.PubClient{}

	ctx := context.WithValue(context.Background(), "test", true)
	upload := &models.Upload{
		UploadId: "upload-id",
		Callback: models.UploadCallback{
			ARN:         "sns-arn",
			Data:        []byte("data"),
			PubsubTopic: givenTopic,
		},
	}
	outputs := []uploader.OutputInfo{{Path: "foo"}}
	backend.sns.(*awsmocks.SNSAPI).On("Publish", mock.Anything).Return(&sns.PublishOutput{}, nil)

	message, err := json.Marshal(models.SNSCallback{
		UploadID: upload.UploadId,
		Data:     upload.Callback.Data,
		Outputs:  outputs,
		Status:   int64(uploader.Status_COMPLETE),
	})
	suite.Require().NoError(err)

	if numFailures > 0 {
		pubsubErr := errors.New("PubSub failure")
		backend.pubclient.(*mocks.PubClient).On("Publish", ctx, []string{expectedTopic}, string(message), (*twitchhttp.ReqOpts)(nil)).Return(pubsubErr).Times(numFailures)
	}
	backend.pubclient.(*mocks.PubClient).On("Publish", ctx, []string{expectedTopic}, string(message), (*twitchhttp.ReqOpts)(nil)).Return(nil).Once()

	err = backend.NotifyCallbacks(ctx, upload, uploader.Status_COMPLETE, outputs)
	suite.Require().NoError(err)
}

func TestMessages(t *testing.T) {
	suite.Run(t, new(MessagesTestSuite))
}
