package backend

import (
	"context"

	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
)

const (
	KeyColumn = "upload_id"
)

type Err error
type UserError struct {
	Err
}

func uploadKey(uploadID string) (map[string]*dynamodb.AttributeValue, error) {
	return dynamodbattribute.MarshalMap(map[string]string{
		KeyColumn: uploadID,
	})
}

func (b *Backend) CreateMetadata(upload models.Upload) error {
	record, err := b.createUploadRecord(upload)
	if err != nil {
		return err
	}

	_, err = b.DynamoDB().PutItem(record)
	if err != nil {
		return errors.Wrap(err, "Dynamo put item error")
	}

	return nil
}

func (b *Backend) GetMetadata(uploadID string) (*models.Upload, error) {
	if uploadID == "" {
		return nil, UserError{errors.New("Get metadata received empty upload id")}
	}

	metadataTable := b.MetadataTable()
	if metadataTable == "" {
		return nil, errors.New("Missing required config argument: METADATA_TABLE")
	}

	key, err := uploadKey(uploadID)
	if err != nil {
		return nil, err
	}

	metadata, err := b.DynamoDB().GetItem(&dynamodb.GetItemInput{
		ConsistentRead: aws.Bool(true),
		Key:            key,
		TableName:      aws.String(metadataTable),
	})
	if err != nil {
		return nil, errors.Wrap(err, "Dynamo get item error")
	}
	if metadata.Item == nil {
		return nil, UserError{errors.New("Upload not found: " + uploadID)}
	}

	result := &models.Upload{}
	err = dynamodbattribute.UnmarshalMap(metadata.Item, result)
	if err != nil {
		return nil, errors.Wrap(err, "Dynamo error unmarshaling metadata")
	}
	return result, nil
}

func (b *Backend) saveStatus(uploadID string, status uploader.Status, statusMessage string) error {
	statusValue := int32(status)
	statusName := uploader.Status_name[statusValue]

	metadataTable := b.MetadataTable()
	if metadataTable == "" {
		return errors.New("Missing required config argument: METADATA_TABLE")
	}

	key, err := uploadKey(uploadID)
	if err != nil {
		return err
	}

	values, err := dynamodbattribute.MarshalMap(map[string]interface{}{
		":sv": aws.Int64(int64(statusValue)),
		":sn": aws.String(statusName),
		":sm": aws.String(statusMessage),
	})
	if err != nil {
		return err
	}

	names := map[string]*string{
		"#sv": aws.String("status_value"),
		"#sn": aws.String("status_name"),
		"#sm": aws.String("status_message"),
	}

	_, err = b.DynamoDB().UpdateItem(&dynamodb.UpdateItemInput{
		Key:                       key,
		TableName:                 aws.String(metadataTable),
		UpdateExpression:          aws.String("SET #sv=:sv, #sn=:sn, #sm=:sm"),
		ExpressionAttributeValues: values,
		ExpressionAttributeNames:  names,
	})

	if err != nil {
		return errors.Wrap(err, "Dynamo update item error")
	}

	return nil
}

func (b *Backend) SetStatus(ctx context.Context, uploadID string, status uploader.Status, statusMessage string) error {
	err := b.saveStatus(uploadID, status, statusMessage)

	return err
}

func (b Backend) createUploadRecord(upload models.Upload) (*dynamodb.PutItemInput, error) {
	tableName := b.MetadataTable()
	if tableName == "" {
		return nil, errors.New("Missing required config argument: METADATA_TABLE")
	}

	item, err := dynamodbattribute.MarshalMap(upload)
	if err != nil {
		return nil, err
	}

	input := dynamodb.PutItemInput{
		TableName: &tableName,
		Item:      item,
	}

	return &input, nil
}
