package backend

import (
	"errors"
	"net/url"
	"strings"
	"testing"

	"code.justin.tv/web/upload-service/awsmocks"
	"code.justin.tv/web/upload-service/models"
	pubclientmocks "code.justin.tv/web/upload-service/pubclient/mocks"
	"code.justin.tv/web/upload-service/rpc/uploader"

	"context"
	"log"

	"code.justin.tv/web/upload-service/transformations"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sns"
	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type MetadataTestSuite struct {
	suite.Suite
	dynamodb  *awsmocks.DynamoDBAPI
	pubclient *pubclientmocks.PubClient

	backend *Backend
	snsArn  *string
}

func (suite *MetadataTestSuite) SetupSuite() {
	logLevel := aws.LogOff
	// Uncomment logLevel for verbose logging from the aws clientlib.
	//logLevel = aws.LogDebugWithRequestErrors
	conf := &aws.Config{
		LogLevel:    &logLevel,                    // Whether or not to print debug messages
		DisableSSL:  aws.Bool(true),               // Moto cannot handle HTTPS
		Endpoint:    aws.String("localhost:5000"), // Connect to moto instead of amazon
		Credentials: credentials.NewStaticCredentials("a", "b", "c"),
		Region:      aws.String(DefaultRegion)}
	be, err := NewBackendFromConfigTableBucket(conf, "metadata_tableName", "s3://bucket/path", false)

	if err != nil {
		log.Fatal(err)
	}

	suite.backend = be
}

func (suite *MetadataTestSuite) SetupTest() {

	suite.dynamodb = &awsmocks.DynamoDBAPI{}
	suite.pubclient = &pubclientmocks.PubClient{}

	out, err := suite.backend.SNS().CreateTopic(&sns.CreateTopicInput{
		Name: aws.String("snsTopic"),
	})
	suite.Require().NoError(err)

	suite.snsArn = out.TopicArn

	input := &dynamodb.CreateTableInput{
		AttributeDefinitions: []*dynamodb.AttributeDefinition{{
			AttributeName: aws.String("upload_id"),
			AttributeType: aws.String("S"),
		}},
		KeySchema: []*dynamodb.KeySchemaElement{{
			AttributeName: aws.String("upload_id"),
			KeyType:       aws.String("HASH"),
		}},
		ProvisionedThroughput: &dynamodb.ProvisionedThroughput{
			ReadCapacityUnits:  aws.Int64(5),
			WriteCapacityUnits: aws.Int64(5),
		},
		TableName: aws.String("metadata_tableName"),
	}
	_, err = suite.backend.DynamoDB().CreateTable(input)
	suite.Require().NoError(err)

}

func (suite *MetadataTestSuite) TearDownTest() {
	_, err := suite.backend.DynamoDB().DeleteTable(&dynamodb.DeleteTableInput{
		TableName: aws.String("metadata_tableName"),
	})
	suite.Require().NoError(err)

	_, err = suite.backend.SNS().DeleteTopic(&sns.DeleteTopicInput{
		TopicArn: suite.snsArn,
	})
	suite.Require().NoError(err)

}

func (suite *MetadataTestSuite) TestCreateMetadataNoTable() {
	backend := Backend{
		dynamodb: suite.dynamodb,
	}

	upload := models.Upload{
		UploadId: "totally-not-a-real-uuid-lel",
	}

	expectedErr := errors.New("Missing required config argument: METADATA_TABLE")

	err := backend.CreateMetadata(upload)

	suite.Require().Equal(expectedErr.Error(), err.Error())
}

func (suite *MetadataTestSuite) TestCreateMetadataWithDynamoError() {
	backend := Backend{
		dynamodb:      suite.dynamodb,
		metadataTable: "totally_not_a_real_table",
	}

	upload := models.Upload{
		UploadId: "totally-not-a-real-uuid-lel",
	}

	expectedErr := errors.New("expected err")

	matchFn := func(arg *dynamodb.PutItemInput) bool {
		return true
	}

	suite.dynamodb.On("PutItem", mock.MatchedBy(matchFn)).Return(nil, expectedErr)

	err := backend.CreateMetadata(upload)

	suite.Require().True(strings.Contains(err.Error(), expectedErr.Error()), "Error did not contain 'expected error': %q", err.Error())
}

func (suite *MetadataTestSuite) TestSetStatus() {
	ctx := context.Background()
	fake_id := "totally-not-a-real-uuid-lel"

	status := uploader.Status_POSTPROCESS_FAILED
	statusMessage := "Message indicating why postprocess failed"

	err := suite.backend.SetStatus(ctx, fake_id, status, statusMessage)

	suite.Require().NoError(err)
}

func (suite *MetadataTestSuite) TestGetUnknownID() {
	fake_id := "totally-not-a-real-uuid-lel"

	gotUpload, err := suite.backend.GetMetadata(fake_id)
	suite.Require().Nil(gotUpload)
	suite.Require().True(strings.Contains(err.Error(), "Upload not found"), "Error did not contain 'Upload not found': %q", err.Error())
}

func (suite *MetadataTestSuite) TestGetEmptyID() {
	fake_id := ""

	gotUpload, err := suite.backend.GetMetadata(fake_id)
	suite.Require().Nil(gotUpload)
	suite.Require().True(strings.Contains(err.Error(), "empty upload id"), "Error did not contain 'empty upload id': %q", err.Error())
}

func (suite *MetadataTestSuite) TestGetIDWhichDoesntExistInDynamo() {
	fake_id := "totally-not-existing"

	fakeGetOutput := &dynamodb.GetItemOutput{
		Item: nil,
	}

	suite.dynamodb.On("GetItem", mock.Anything).Return(fakeGetOutput, nil)

	gotUpload, err := suite.backend.GetMetadata(fake_id)
	suite.Require().Nil(gotUpload)
	suite.Require().True(strings.Contains(err.Error(), "Upload not found:"), "Error did not contain 'Upload not found:': %q", err.Error())
}

func (suite *MetadataTestSuite) TestTransformSerialization() {
	upload := models.Upload{
		UploadId: uuid.NewV4().String(),
		Outputs: []models.Output{
			{
				Transformations: []transformations.Transformation{
					&transformations.AspectRatio{2.0},
					&transformations.Crop{10, 11, 12, 13},
					&transformations.MaxHeight{14},
					&transformations.MaxWidth{15},
					&transformations.ResizeDimensions{16, 17},
					&transformations.ResizeDimensions{Height: 18},
					&transformations.ResizeDimensions{Width: 19},
					&transformations.ResizePercentage{75},
					&transformations.Transcode{Format: "png"},
					&transformations.Transcode{Format: "jpeg", Quality: 80},
				},
				Name: "test",
			},
		},
	}

	err := suite.backend.CreateMetadata(upload)
	suite.Require().NoError(err)

	gotUpload, err := suite.backend.GetMetadata(upload.UploadId)
	suite.Require().NoError(err)

	suite.Require().Equal(upload, *gotUpload)
}

func (suite *MetadataTestSuite) TestCreatePresignedUrl() {
	signedUrl, err := suite.backend.CreatePresignedUrl("UPLOAD-ID")
	suite.Require().NoError(err)

	_, err = url.Parse(signedUrl)
	suite.Require().NoError(err)
}

func (suite *MetadataTestSuite) TestFullMetadataIntegration() {
	fake_id := "totally-not-a-real-uuid-lel"

	upload := models.Upload{
		UploadId:      fake_id,
		StatusValue:   int32(0),
		StatusName:    "REQUESTED",
		StatusMessage: "",
	}
	err := suite.backend.CreateMetadata(upload)
	suite.Require().NoError(err)

	gotUpload1, err := suite.backend.GetMetadata(fake_id)
	suite.Require().NoError(err)
	suite.Require().Equal(int32(0), gotUpload1.StatusValue)
	suite.Require().Equal("REQUESTED", gotUpload1.StatusName)
	suite.Require().Equal("", gotUpload1.StatusMessage)

	status := uploader.Status_POSTPROCESS_FAILED
	statusValue := int32(uploader.Status_POSTPROCESS_FAILED)
	statusName := uploader.Status_name[statusValue]
	statusMessage := "Message indicating why postprocess failed"
	ctx := context.Background()

	suite.pubclient.On("Publish", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil)

	err = suite.backend.SetStatus(ctx, fake_id, status, statusMessage)
	suite.Require().NoError(err)

	gotUpload2, err := suite.backend.GetMetadata(fake_id)
	suite.Require().NoError(err)
	suite.Require().Equal(statusValue, gotUpload2.StatusValue)
	suite.Require().Equal(statusName, gotUpload2.StatusName)
	suite.Require().Equal(statusMessage, gotUpload2.StatusMessage)
}

func TestMetadata(t *testing.T) {
	suite.Run(t, new(MetadataTestSuite))
}
