package main

import (
	"context"
	"fmt"
	"log"
	"net/http"

	"bytes"
	"io/ioutil"

	"time"

	"errors"
	"strconv"

	"encoding/json"

	"code.justin.tv/common/config"
	"code.justin.tv/common/golibs/pubsubclient"
	"code.justin.tv/web/upload-service/backend"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
)

var (
	EndpointUrl = ""
	SNSARN      = ""
	serviceName = ""
)

func init() {
	config.Register(map[string]string{
		"SERVICE_NAME":        "",
		"PUBSUB_TOPIC_PREFIX": "pubsubtest.upload.",
		"PUBSUB_URL":          "wss://pubsub-edge-darklaunch.twitch.tv:443/v1",
		"AWS_ID":              "",
	})
}

func main() {
	err := config.Parse()
	if err != nil {
		log.Fatalf("Failed to parse config: %v", err)
	}

	serviceName = config.Resolve("SERVICE_NAME")
	EndpointUrl = "http://" + serviceName + "." + config.Environment() + ".us-west2.twitch.tv/"
	SNSARN = "arn:aws:sns:us-west-2:" + config.Resolve("AWS_ID") + ":" + serviceName + "-callback"

	fmt.Printf("Beginning test:\nEndpoint:%q\nSNS ARN:%q\n", EndpointUrl, SNSARN)

	testUploadWithPolling()
	testUploadWithPubSubListener()
}

func testUploadWithPolling() {
	startTime := time.Now()
	signedUrl, uploadId := getSignedUrl(backend.Empty)

	originalBytes := uploadImage(signedUrl)

	status := pollForStatusChange(uploadId)

	if status == uploader.Status_COMPLETE {
		validateUploadedImage(originalBytes, startTime)
	} else {
		log.Fatal("Upload status never changed to COMPLETE after 20 seconds")
	}
}

func testUploadWithPubSubListener() {
	startTime := time.Now()
	signedUrl, uploadId := getSignedUrl(backend.Default)

	sawCompleteStatus, originalBytes := listenForStatusChange(uploadId, uploader.Status_POSTPROCESS_COMPLETE, signedUrl)

	if sawCompleteStatus {
		validateUploadedImage(originalBytes, startTime)
	} else {
		log.Fatal("Upload status never changed to COMPLETE after 20 seconds")
	}
}

func validateUploadedImage(originalBytes int, startTime time.Time) {
	url := "https://s3-us-west-2.amazonaws.com/" + serviceName + "-output/jn-homepage-pics/me.jpg"
	s3HttpClient := &http.Client{}
	res, err := s3HttpClient.Head(url)
	if err != nil || res.StatusCode != 200 {
		log.Fatal("Upload status changed to COMPLETE but object could not be accessed")
	}
	resBytes, err := strconv.Atoi(res.Header.Get("Content-Length"))
	if err != nil || resBytes >= int(float64(originalBytes)*0.75) {
		log.Fatal("No content-length header or image not resized", res.Header.Get("Content-Length"), resBytes, originalBytes)
	}
	fmt.Println("test succeeded after " + time.Now().Sub(startTime).String())
	return
}

func listenForStatusChange(uploadId string, expectedStatus uploader.Status, signedUrl string) (bool, int) {
	pubsubUrl := config.Resolve("PUBSUB_URL")
	topicPrefix := config.Resolve("PUBSUB_TOPIC_PREFIX")

	log.Printf("Initializing pubsub with:\nUrl:%q\nPrefix:%q\n", pubsubUrl, topicPrefix)

	c := pubsubclient.New(pubsubUrl, nil)

	topic := topicPrefix + uploadId

	log.Printf("Subscribing to topic: %q\n", topic)

	err := c.Subscribe(topic, "")
	if err != nil {
		log.Fatalf("Error subscribing to topic %q:\n%v", topic, err)
	}

	originalBytes := uploadImage(signedUrl)

	for i := 0; i < 10; i++ {
		log.Println("Listening for message")
		msg, err := getMessage(c, 20*time.Second)

		if err != nil {
			log.Fatal(err)
		}

		bytes := []byte(msg.Message)
		req := &models.SNSCallback{}
		if err := json.Unmarshal(bytes, req); err != nil {
			log.Fatal(err)
		}

		if uploader.Status(req.Status) == expectedStatus {
			return true, originalBytes
		}
	}

	return false, originalBytes
}

func getMessage(c pubsubclient.Client, timeout time.Duration) (*pubsubclient.Message, error) {
	var (
		msg *pubsubclient.Message
		err error
	)

	ch := make(chan struct{})
	go func() {
		msg, err = c.NextMessage()
		if msg != nil {
			fmt.Println("got message", msg.Message)
		}
		if err != nil {
			fmt.Printf("got error %v\n", err)
		}
		close(ch)
	}()
	select {
	case <-ch:
	case <-time.After(timeout):
		return nil, errors.New("timed out waiting for message")
	}
	return msg, err
}

func pollForStatusChange(uploadId string) uploader.Status {
	var status uploader.Status

	for i := 0; i < 200 && status != uploader.Status_COMPLETE; i++ {
		status = checkStatus(uploadId)
		niceStatus := uploader.Status_name[int32(status)]
		fmt.Printf("[%s] Upload "+uploadId+" has status: "+niceStatus+"\n", time.Now().String())
		time.Sleep(100 * time.Millisecond)
	}

	return status
}

func getSignedUrl(pubsubTopic string) (string, string) {
	client := uploader.NewUploaderProtobufClient(EndpointUrl, &http.Client{})

	callback := &uploader.Callback{
		SnsTopicArn: SNSARN,
		PubsubTopic: pubsubTopic,
	}
	monitoring := &uploader.Monitoring{}
	transform := &transformations.ResizePercentage{75}

	request := &uploader.UploadRequest{
		Outputs: []*uploader.Output{
			{
				Transformations: []*uploader.Transformation{transform.AsProto()},
				PostValidation:  &uploader.Validation{},
				Name:            "me.jpg",
			},
		},
		PreValidation: &uploader.Validation{},
		OutputPrefix:  "s3://" + serviceName + "-output/jn-homepage-pics/",
		Callback:      callback,
		Monitoring:    monitoring,
	}

	signedUrl, err := client.Create(context.Background(), request)

	if err != nil {
		log.Fatal(err)
	} else {
		fmt.Println("got url", signedUrl)
	}

	return signedUrl.Url, signedUrl.UploadId
}

func uploadImage(signedUrl string) int {
	inputImage, err := ioutil.ReadFile("me.jpg")

	if err != nil {
		log.Fatal(err)
	}

	s3HttpClient := &http.Client{}
	req, err := http.NewRequest(http.MethodPut, signedUrl, bytes.NewBuffer(inputImage))

	if err != nil {
		log.Fatal(err)
	}

	resp, reqErr := s3HttpClient.Do(req)

	if reqErr != nil {
		log.Fatal(reqErr)
	}

	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		body, _ := ioutil.ReadAll(resp.Body)

		fmt.Println(string(body))
		log.Fatal("Did not receive a success response from S3")
	}

	return len(inputImage)
}

func checkStatus(uploadId string) uploader.Status {
	client := uploader.NewUploaderProtobufClient(EndpointUrl, &http.Client{})

	statusRequest := &uploader.StatusRequest{
		UploadId: uploadId,
	}

	statusResponse, err := client.Status(context.Background(), statusRequest)

	if err != nil {
		log.Fatal(err)
	}

	return statusResponse.Status
}
