package main

import (
	builtinLog "log"

	"code.justin.tv/common/config"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/web/upload-service/api"
	"code.justin.tv/web/upload-service/backend"

	"os"

	"code.justin.tv/web/upload-service/transformations/cwrapper"
	log "github.com/sirupsen/logrus"
)

const DefaultLogLevel = log.InfoLevel

func init() {
	config.Register(map[string]string{
		"LOG_LEVEL": DefaultLogLevel.String(),
	})
}

func configureLogging() {
	logPath := "/var/log/upload-service.log"
	logFile, err := os.OpenFile(logPath, os.O_CREATE|os.O_APPEND|os.O_WRONLY, 0755)
	if err != nil {
		builtinLog.Fatalf("Could not open log file %s: %v", logPath, err)
	}
	log.SetOutput(logFile)

	log.SetFormatter(&log.JSONFormatter{})
	level, err := log.ParseLevel(config.Resolve("LOG_LEVEL"))
	if err != nil {
		level = DefaultLogLevel
		log.WithFields(map[string]interface{}{
			"defaultLevel": DefaultLogLevel.String(),
			"configLevel":  config.Resolve("LOG_LEVEL"),
		}).WithError(err).Warn("Error parsing log level, using default")
	}
	log.SetLevel(level)
}

func main() {
	// To add app-specific config flags, you can call config.Register()
	// https://godoc.internal.justin.tv/code.justin.tv/common/config
	err := config.Parse()
	if err != nil {
		builtinLog.Fatalf("Failed to parse config: %v", err)
	}

	configureLogging()

	transformer := &cwrapper.IMagickTransformer{}
	transformer.Initialize()
	defer transformer.Terminate()

	be, err := backend.NewBackend()

	if err != nil {
		log.Fatal(err)
	}

	server, err := api.NewServer(twitchserver.NewServer(), be, config.ClusterStatsd(), transformer)
	if err != nil {
		log.WithError(err).Fatal("Failed to create the server")
	}

	log.WithError(twitchserver.ListenAndServe(server, nil)).Fatal("Server exited unexpectedly")
}
