package middleware

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/web/upload-service/rpc/uploader"

	"github.com/cactus/go-statsd-client/statsd"
	log "github.com/sirupsen/logrus"
)

func StatsMiddleware(instanceID string, stats statsd.Statter) func(http.Handler) http.Handler {
	// This function inspired by: https://git-aws.internal.justin.tv/edge/visage/blob/master/internal/tracking/statsd.go
	return func(h http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			statusWriter := &statusLenResponseWriter{w, 0, 200}
			startTime := time.Now()

			h.ServeHTTP(statusWriter, r)

			duration := time.Since(startTime)
			responseCode := statusWriter.status

			endpoint := r.URL.Path
			if strings.HasPrefix(r.URL.Path, uploader.UploaderPathPrefix) {
				endpoint = r.URL.Path[len(uploader.UploaderPathPrefix):]
			}
			statName := metricName(instanceID, r.Method, endpoint, responseCode)

			err := stats.TimingDuration(statName, duration, 1)
			if err != nil {
				log.WithFields(log.Fields{
					"statName": statName,
				}).WithError(err).Warn("Error publishing to statsd")
			}
		})
	}
}

func metricName(instanceID, method, endpoint string, responseCode int) string {
	// This function inspired by: https://git-aws.internal.justin.tv/edge/visage/blob/master/internal/tracking/statsd.go

	// Replace pattern characters like / and : with graphite-friendly alternatives:
	// - /feeds/recent -> feeds-recent
	// - /feeds/:feed_id/posts -> feeds-_feed_id-posts
	endpoint = strings.Trim(endpoint, "/")
	endpoint = strings.Replace(endpoint, "/", "-", -1)
	endpoint = strings.Replace(endpoint, ":", "_", -1)

	return fmt.Sprintf("%s.%s.%s.%d", instanceID, method, endpoint, responseCode)
}
